/**
 * Copyright 2020 (C) Hailo Technologies Ltd.
 * All rights reserved.
 *
 * Hailo Technologies Ltd. ("Hailo") disclaims any warranties, including, but not limited to,
 * the implied warranties of merchantability and fitness for a particular purpose.
 * This software is provided on an "AS IS" basis, and Hailo has no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 * You may use this software in the development of any project.
 * You shall not reproduce, modify or distribute this software without prior written permission.
 **/
/**
 * @ file example.c
 * This example demonstrates the basic data-path on HailoRT.
 * The program scans for Hailo-8 devices connected to a provided Ethernet interface, generates random dataset,
 * and runs it through the device.
 **/
#include <net/if.h>
#include <pthread.h>
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <linux/limits.h>
#include <time.h>

#include <hailo/hailort.h>
#include "example_utils.h"

#define NOF_STREAMS (15)
#define NSEC_IN_SEC (1e+9)
#define LATENCY_MEASUREMENTS (100)
#define DEFAULT_POWER_MEASUREMENT_DELAY_MS (1100 / 1000.0 * 256 * 2 * 1.2)

struct timespec sent_clock_t[LATENCY_MEASUREMENTS];
struct timespec recv_clock_t[NOF_STREAMS-1][LATENCY_MEASUREMENTS];
unsigned int actual_measurments;

void* _send_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    write_thread_args_t *write_args = (write_thread_args_t*)args;
    uint32_t input_size = write_args->frame_hw_size;
    uint8_t* src_data = (uint8_t*)malloc(input_size);
    unsigned lat_counter = 0;
    uint32_t flag_100 = 0;
    struct timespec ts;
    float delay = 0.0;

    printf("-I- Send Thread stated TX frame size:%d\n", input_size);
    if (NULL == src_data) {
        printf("-E- Failed to allocate buffers\n");
        status = HAILO_OUT_OF_HOST_MEMORY;
        goto l_release_buffers;
    }
    for(size_t i = 0; i < input_size; i++) {
        src_data[i] = (uint8_t)(rand() % 256);
    }
    flag_100 = (uint32_t)write_args->num_images / 10;
    if (flag_100==0)
        flag_100 = 1;
    
    if (write_args->source_fps > 0) {
        delay = 1000000.0 / write_args->source_fps;
        printf(CYAN);
        printf("-I- Setting the delay to %f usec\n", delay);
        printf(RESET);
    }
    for (uint32_t i = 1; i <= (uint32_t)write_args->num_images; i++) {
        if (delay > 0.0)
            usleep(delay);
        if ((i % flag_100==0) && (lat_counter < LATENCY_MEASUREMENTS)) {
            clock_gettime(CLOCK_REALTIME, &ts);
            printf("-I- [%10ld.%ld s] Send frame [%3d/%3d] 0x%x%x%x%x\n", (long)ts.tv_sec, ts.tv_nsec/1000000, i, write_args->num_images, src_data[0], src_data[1], src_data[2], src_data[3]);
            sent_clock_t[lat_counter++] = ts;
        }
        status = hailo_stream_sync_write_all_raw_buffer(write_args->input_stream, src_data, 0, input_size);
        if (status != HAILO_SUCCESS) {
            printf("-E- hailo_stream_sync_write_all_raw_buffer failed: %d\n",status);
            break;
        }
    }

l_release_buffers:
    FREE(src_data);

    write_args->status = status;
    return NULL;
}

double get_time_from_ts(struct timespec ts) {
    double result = (double)(ts.tv_sec * 1000 + (double)(ts.tv_nsec / 1000000.0));
    return result;
}

void* _recv_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    recv_thread_args_t *recv_args = (recv_thread_args_t *)args;
    uint32_t datasize = recv_args->frame_hw_size;
    struct timespec ts;
    uint8_t *dst_data = (uint8_t*)malloc(datasize);
    unsigned lat_counter = 0;
    uint32_t flag_100 = 0;
    
    printf("-I- Recv thread %d started, datasize: %d\n", recv_args->tid, datasize);

    flag_100 = (uint32_t)recv_args->num_images / 10;
    if (flag_100==0)
        flag_100 = 1;
    
    actual_measurments = 0;

    for (uint32_t j = 1; j <= (uint32_t)recv_args->num_images; j++) {
        status = hailo_stream_sync_read_all_raw_buffer(recv_args->output_stream, dst_data, 0, datasize);
        REQUIRE_SUCCESS(status, l_exit, "hailo_stream_sync_read_all_raw_buffer failed");
        
        if ((j % flag_100==0) && (lat_counter < LATENCY_MEASUREMENTS)) {
            clock_gettime(CLOCK_REALTIME, &ts);
            printf("-I- [%10ld.%ld s] TID:%d Recv [%3d/%3d] 0x%x%x%x%x\n", (long)ts.tv_sec, ts.tv_nsec/1000000, recv_args->tid, j, recv_args->num_images, dst_data[0], dst_data[1], dst_data[2], dst_data[3]);
            recv_clock_t[recv_args->tid][lat_counter++] = ts;
            actual_measurments++;
        }
    }
    
l_exit:
    FREE(dst_data);
    recv_args->status = status;
    return NULL;
}

hailo_status infer(
    hailo_input_stream input_stream, hailo_output_stream *output_streams, hailo_stream_info_t *all_strems_info,
    int output_stream_cnt, int num_images, int write_log, size_t* host_output_frame_size, size_t host_input_frame_size, int source_fps)
{
    hailo_status status = HAILO_SUCCESS;
    pthread_t write_thread = 1;
    recv_thread_args_t recv_args[NOF_STREAMS-1];
    pthread_t recv_threads[NOF_STREAMS-1];

    // struct timespec st, et;
    // uint8_t* src_data = (uint8_t*)malloc(all_strems_info[0].hw_frame_size);
    // uint8_t* dst_data = (uint8_t*)malloc(all_strems_info[1].hw_frame_size);

    write_thread_args_t write_args = {
        .input_stream = input_stream,
        .status = HAILO_UNINITIALIZED,
        .output_streams_cnt = output_stream_cnt,
        .num_images = num_images,
        .host_frame_size = host_input_frame_size,
        .frame_hw_size = all_strems_info[0].hw_frame_size,
        .source_fps = source_fps
    };
    for (int ii=0; ii<output_stream_cnt; ii++) {
        recv_args[ii].output_stream = output_streams[ii];
        recv_args[ii].tid = ii;
        recv_args[ii].status = HAILO_UNINITIALIZED;
        recv_args[ii].num_images = num_images;
        recv_args[ii].write_log = write_log;
        recv_args[ii].host_frame_size = host_output_frame_size[ii];
        recv_args[ii].frame_hw_size = all_strems_info[ii+1].hw_frame_size;
        (void) pthread_create(&recv_threads[ii], NULL, _recv_thread, &recv_args[ii]);
    };

    (void) pthread_create(&write_thread, NULL, _send_thread, &write_args);
    pthread_join(write_thread, NULL);

    if (HAILO_SUCCESS != write_args.status) {
        printf("-E- write_thread failed\n");
        status = HAILO_INTERNAL_FAILURE;
    }
    for (int ii=0; ii<output_stream_cnt; ii++) {
        pthread_join(recv_threads[ii], NULL);
        printf("-I- Closing Recv thread #%d: %d\n", recv_args[ii].tid, recv_args[ii].status);
        if (HAILO_SUCCESS != recv_args[ii].status) {
            printf("-E- Recv thread %d failed\n", recv_args[ii].tid);
            status = HAILO_INTERNAL_FAILURE;
        }
    }
    // for (uint32_t i = 1; i <= (uint32_t)num_images; i++) {
    //     clock_gettime(CLOCK_REALTIME, &st);
    //     status = hailo_stream_sync_write_all_raw_buffer(input_stream, src_data, 0, all_strems_info[0].hw_frame_size);
    //     status = hailo_stream_sync_read_all_raw_buffer(output_streams[0], dst_data, 0, all_strems_info[1].hw_frame_size);
    //     clock_gettime(CLOCK_REALTIME, &et);
    //     printf("-I- [%f ms] Infer [%3d/%3d]\n", get_time_from_ts(et)-get_time_from_ts(st), i, num_images);

    // }
    // FREE(src_data);
    // FREE(dst_data);
    
    return status;
}

void print_net_banner(hailo_input_stream_params_by_name_t* inputs, hailo_output_stream_params_by_name_t* outputs, size_t in_cnt, size_t out_cnt) 
{
    printf(MAGENTA);
    printf("-I-----------------------------------------------------------\n");
    printf("-I- Dir Name                  Transform Type    Flags Order\n");
    printf("-I-----------------------------------------------------------\n");
    for (int i=0; i<in_cnt; i++) {
        printf("-I- IN: %-20.20s: %-9.9s %-7.7s %-5.5s %-4.4s\n", 
            inputs[i].name, 
            get_transform_string(inputs[i].params.pcie_params.base_params.transform_mode), 
            get_type_string(inputs[i].params.pcie_params.base_params.user_buffer_format.type),
            get_flags_string(inputs[i].params.pcie_params.base_params.user_buffer_format.flags),
            get_order_string(inputs[i].params.pcie_params.base_params.user_buffer_format.order));
    }
    printf("-I-----------------------------------------------------------\n");
    for (int i=0; i<out_cnt; i++) {
        printf("-I- OUT:%-20.20s: %-9.9s %-7.7s %-5.5s %-4.4s\n", 
            outputs[i].name, 
            get_transform_string(outputs[i].params.pcie_params.base_params.transform_mode), 
            get_type_string(outputs[i].params.pcie_params.base_params.user_buffer_format.type),
            get_flags_string(outputs[i].params.pcie_params.base_params.user_buffer_format.flags),
            get_order_string(outputs[i].params.pcie_params.base_params.user_buffer_format.order));
    }
    printf("-I-----------------------------------------------------------\n");
    printf(RESET);
}

double calc_latency(int count, double precentile, int write_log) {
    double cur_rcv;
    double cur_snd;
    double latencies[actual_measurments];

    for (int j=0; j<actual_measurments; j++) {
        cur_snd = get_time_from_ts(sent_clock_t[j]);
        cur_rcv = get_time_from_ts(recv_clock_t[0][j]);
        for (int ii=1; ii<count; ii++) {
            if (get_time_from_ts(recv_clock_t[ii][j]) > cur_rcv) {
                cur_rcv = get_time_from_ts(recv_clock_t[ii][j]);
            }
        }
        latencies[j] = (cur_rcv - cur_snd);
    }

    if ((precentile<0) || (precentile>1)) {
        precentile = 0.5;
    }
    if (write_log>0) {
        FILE *fp = fopen("latency_redouts.txt", "w");
        for (int ii=0; ii<actual_measurments; ii++) {
            fprintf(fp, "%f\n",latencies[ii]);
        }
        fclose(fp);
    }
    return latencies[(int)(precentile * actual_measurments)];
}

void print_inference_stats(struct timespec start_time, struct timespec end_time, hailo_stream_info_t *all_stream_infos, int output_stream_cnt, 
    int num_images, uint32_t latency_readout, int write_log, hailo_latency_measurement_result_t driver_lat, int nbatch)
{
    double start_time_secs = (double)start_time.tv_sec + ((double)start_time.tv_nsec / NSEC_IN_SEC);
    double end_time_secs = (double)end_time.tv_sec + ((double)end_time.tv_nsec / NSEC_IN_SEC);
    double infer_time_secs = end_time_secs - start_time_secs;
    static float mbit_per_byte = 8.0f / 1024.0f / 1024.0f;
    uint32_t send_frame_size = all_stream_infos[0].hw_frame_size;
    uint32_t recv_frame_size = 0;

    printf(BOLDGREEN);
    printf("-I-----------------------------------------------\n");
    printf("-I- Batch Size:      %-4d\n", nbatch);
    printf("-I- Total time:      %4.2lf sec\n", infer_time_secs);
    printf("-I- Average FPS:     %4.2lf\n", num_images / infer_time_secs);
    printf("-I- 95p Latency:     %3.2lf ms\n", calc_latency(output_stream_cnt, 0.95, write_log));
    printf("-I- Driver Latency:  %3.2lf ms\n", driver_lat.avg_hw_latency_ms);
    printf("-I- Send data rate:  %-4.2lf Mbit/s\n",
        (double)(num_images) * send_frame_size * mbit_per_byte / infer_time_secs);
    for (int i=1; i<=output_stream_cnt; i++) {
        recv_frame_size = all_stream_infos[i].hw_frame_size;
        printf("-I- Recv[%d] data rate: %-4.2lf Mbit/s\n", i,
            (double)(num_images) * recv_frame_size * mbit_per_byte / infer_time_secs);
        printf("-I-----------------------------------------------\n");
    }
    printf(RESET);
}

void set_default_eth_output_stream_params(hailo_eth_output_stream_params_t* stream) {
    stream->base_params.transform_mode = HAILO_STREAM_TRANSFORM_COPY;
    stream->base_params.user_buffer_format.order = HAILO_FORMAT_ORDER_NHWC;
    stream->base_params.user_buffer_format.flags = HAILO_FORMAT_FLAGS_QUANTIZE;
    stream->base_params.user_buffer_format.type = HAILO_FORMAT_TYPE_UINT8;
    stream->device_port = 0;
    stream->host_address.sin_family = AF_INET;
    stream->host_address.sin_port = 0;
    stream->host_address.sin_addr.s_addr = INADDR_ANY;
    stream->base_params.buffers_threshold = 1;
    // stream->is_sync_enabled = true;
    memset(&stream->host_address.sin_zero, 0, sizeof(stream->host_address.sin_zero));
}

void set_default_pcie_output_stream_params(hailo_pcie_output_stream_params_t* stream) {
    stream->base_params.transform_mode = HAILO_STREAM_TRANSFORM_COPY;
    stream->base_params.user_buffer_format.order = HAILO_FORMAT_ORDER_NHWC;
    stream->base_params.user_buffer_format.flags = HAILO_FORMAT_FLAGS_QUANTIZE;
    stream->base_params.user_buffer_format.type = HAILO_FORMAT_TYPE_UINT8;
}

hailo_status create_and_load_hef(hailo_device device, hailo_hef *hef, char* hef_file, hailo_network_group *network_group, int nbatch)
{
    hailo_status status = HAILO_SUCCESS;
    size_t number_of_network_groups = 1;

    status = hailo_create_hef_file(hef, hef_file);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create hef file");
    printf(MAGENTA);
    printf("-I- Loading HEF file from %s\n", hef_file);
    printf(RESET);

    const char* net_group_name = "yolov4";
    hailo_configure_params_t configure_params = {0};
    configure_params.network_group_params_count = 1;
    configure_params.network_group_params[0].batch_size = nbatch;
    memcpy(configure_params.network_group_params[0].name, net_group_name, 6);
 
    status = hailo_configure_device(device, *hef, &configure_params, network_group, &number_of_network_groups);

    REQUIRE_SUCCESS(status, l_exit, "Failed to configure device from hef");
    if (number_of_network_groups > 1) {
        status = HAILO_UNINITIALIZED;
        printf("-E- Got network_group=%ld", number_of_network_groups);
        REQUIRE_SUCCESS(status, l_exit, "Failed to extract network group, larger than 1");
    }
    
l_exit:
    return status;
}

hailo_status activate_network_group(hailo_hef *hef, hailo_stream_info_t *streams_info, hailo_network_group network_group, hailo_activated_network_group *active_net_g, hailo_input_stream_params_by_name_t *input_stream_params, hailo_output_stream_params_by_name_t *output_stream_param, size_t *output_stream_cnt)
{
    hailo_status status = HAILO_SUCCESS;
    hailo_network_group_params_t network_group_params = {0};//HAILO_NETWORK_GROUP_PARAMS_DEFAULT;
    size_t out_param_size = NOF_STREAMS-1;
    size_t in_param_size  = 1;
    size_t number_of_streams;
    hailo_stream_transform_mode_t transform_mode = HAILO_STREAM_NO_TRANSFORM;// HAILO_STREAM_NO_TRANSFORM

    status = hailo_hef_get_all_stream_infos(hef, NULL, streams_info, sizeof(*streams_info), &number_of_streams);

    status = hailo_make_input_stream_params(network_group, transform_mode, true, HAILO_FORMAT_TYPE_UINT8, input_stream_params, &in_param_size);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create input_params");
    
    status = hailo_make_output_stream_params(network_group, transform_mode, true, HAILO_FORMAT_TYPE_UINT8, output_stream_param, &out_param_size);
    *output_stream_cnt = out_param_size;
    REQUIRE_SUCCESS(status, l_exit, "Failed to create output_params");    
    
    if (output_stream_param->params.pcie_params.base_params.user_buffer_format.order != HAILO_FORMAT_ORDER_HAILO_NMS)
        network_group_params.latency = HAILO_LATENCY_MEASURE;
    output_stream_param->params.eth_params.base_params.buffers_threshold = 1;
    input_stream_params->params.eth_params.base_params.buffers_threshold = 1;
    
    // input_stream_params->params.pcie_params.base_params.user_buffer_format.flags = streams_info[0].format.flags;
    // input_stream_params->params.pcie_params.base_params.user_buffer_format.flags |= HAILO_FORMAT_FLAGS_TRANSPOSED;
    
    print_net_banner(input_stream_params, output_stream_param, in_param_size, out_param_size);
    
    status = hailo_activate_network_group(network_group, 
                                          &network_group_params, 
                                          input_stream_params, // Input params
                                          in_param_size,       // number of inputs
                                          output_stream_param, // Output params
                                          out_param_size,      // number of ouputs
                                          active_net_g);
    REQUIRE_SUCCESS(status, l_exit, "Failed to activate network group");    

l_exit:
    return status;
}

hailo_status initialize_eth_device(const char *interface_name, hailo_device *device)
{
    hailo_status status = HAILO_SUCCESS;
    hailo_eth_device_info_t eth_device_info = {0};
    size_t number_of_devices = 0;
    
    status = hailo_scan_ethernet_devices(interface_name, &eth_device_info, 1, &number_of_devices, HAILO_DEFAULT_ETH_SCAN_TIMEOUT_MS);
    REQUIRE_SUCCESS(status, l_exit, "Failed to scan for eth_devices");
    if (0 == number_of_devices) {
        printf("No eth_device found on the given interface\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_exit;
    }

    status = hailo_create_ethernet_device(&eth_device_info, device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create eth_device");

l_exit:
    return status;
}

hailo_status initialize_pcie_device(hailo_pcie_device_info_t *pcie_device_info, hailo_device *device)
{
    hailo_status status = HAILO_UNINITIALIZED;
    size_t number_of_devices = 0;

    status = hailo_scan_pcie_devices(pcie_device_info, 4, &number_of_devices);
    REQUIRE_SUCCESS(status, l_exit, "Failed to scan for pcie_devices");

    if (0 == number_of_devices) {
        printf("No pcie_device found\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_exit;
    }

    status = hailo_create_pcie_device(pcie_device_info, device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create pcie_device");

    status = HAILO_SUCCESS;
l_exit:
    return status;
}

hailo_status initiazlie_inference(hailo_device *device, hailo_input_stream *input_stream, hailo_stream_info_t *streams_info,
    hailo_output_stream *output_streams, size_t *host_input_frame_size, size_t *host_output_frame_size, char* hef_path, size_t* output_stream_cnt, 
    hailo_activated_network_group *active_net_g, const char *interface, int nbatch)
{
    // NADAV: Added two devices to support multiple M.2 connected to the machine
    hailo_pcie_device_info_t pcie_device_info[4];
    hailo_hef hef = NULL;
    hailo_status status = HAILO_SUCCESS;
    hailo_network_group network_groups = NULL;
    hailo_input_stream_params_by_name_t input_stream_params = {0};
    hailo_output_stream_params_by_name_t output_stream_params[NOF_STREAMS];

    if (strncmp("pcie", interface, 4) == 0) {
        status = initialize_pcie_device(pcie_device_info, device);
        REQUIRE_SUCCESS(status, l_exit, "Failed to initialize PCIe device");
    } else {
        status = initialize_eth_device(interface, device);
        REQUIRE_SUCCESS(status, l_exit, "Failed to initialize Eth device");       
    }
    status = create_and_load_hef(*device, &hef, hef_path, &network_groups, nbatch);

    REQUIRE_SUCCESS(status, l_exit, "Failed to create and load HEF");
    
    status = activate_network_group(hef, streams_info, network_groups, active_net_g, &input_stream_params, output_stream_params, output_stream_cnt);
    REQUIRE_SUCCESS(status, l_release_hef, "Failed to activate network group and streams");
    
    REQUIRE_SUCCESS(status, l_release_hef, "Failed to create and activate pcie_streams");

    status = hailo_get_input_stream_by_name(*active_net_g, streams_info[0].name, input_stream);
    REQUIRE_SUCCESS(status, l_release_hef, "Failed getting stream info for input");

    *host_input_frame_size = hailo_get_input_stream_frame_size(*input_stream);        
    for (size_t i = 0; i < *output_stream_cnt; i++) {
        status = hailo_get_output_stream_by_name(*active_net_g, streams_info[i+1].name, &output_streams[i]);
        REQUIRE_SUCCESS(status, l_release_hef, "Failed getting stream info for output");
        host_output_frame_size[i] = hailo_get_output_stream_frame_size(output_streams[i]);
    }

l_release_hef:
    (void) hailo_release_hef(hef);

l_exit:
    return status;
}

hailo_status print_power_stats(hailo_device device) {
    hailo_status status;
    hailo_power_measurement_data_t measurement_data[3];

    status = hailo_stop_power_measurement(device);
    status = hailo_get_power_measurement(device, 0, true, &measurement_data[0]);
    printf(BOLDBLUE);
    printf("-I-----------------------------------------------\n");
    if (measurement_data[0].total_number_of_samples==0) {
        printf("-W- The test was too short to measure power\n");
    } else {
        printf("-I- Total samples: %d ", measurement_data[0].total_number_of_samples);
        printf(" Average time per sample (ms): %.3f\n", measurement_data[0].average_time_value_milliseconds);
        printf("-I- CORE  [%.3f, %.3f] W\n", measurement_data[0].min_value, measurement_data[0].max_value);
    }
    status = hailo_get_power_measurement(device, 1, true, &measurement_data[1]);
    if (measurement_data[1].total_number_of_samples==0) {
        printf("-W- The test was too short to measure power\n");
    } else {
        printf("-I- TOP   [%.3f, %.3f] W\n", measurement_data[1].min_value, measurement_data[1].max_value);
    }
    status = hailo_get_power_measurement(device, 2, true, &measurement_data[2]);
    if (measurement_data[2].total_number_of_samples==0) {
        printf("-W- The test was too short to measure power\n");
    } else {
        printf("-I- MIPI  [%.3f, %.3f] W\n", measurement_data[2].min_value, measurement_data[2].max_value);
    }
    float min=0;
    float max=0;
    for (int i=0;i<3;i++) {
        min += measurement_data[i].min_value;
        max += measurement_data[i].max_value;
    }
    printf("-I- TOTAL [%.3f, %.3f] W\n", min, max);
    printf(RESET);
    return status;
}

int main(int argc, char **argv)
{
    hailo_status status = HAILO_SUCCESS;
    hailo_device device = NULL;
    hailo_input_stream input_stream = NULL;
    hailo_output_stream output_streams[NOF_STREAMS];
    hailo_stream_info_t all_stream_infos[NOF_STREAMS];
    size_t host_input_frame_size = 0;
    size_t host_output_frame_size[NOF_STREAMS];
    struct timespec start_time = {0};
    struct timespec end_time = {0};
    size_t output_stream_cnt = 0;
    char *config = "";
    int opt;
    char *iface = "pcie";
    int num_img = 100;
    int nbatch = 1;
    int write_log = 0;
    int source_fps = 0;
    // hailo_dvm_options_t dvm_option = HAILO_DVM_OPTIONS_OVERCURRENT_PROTECTION;
    hailo_dvm_options_t dvm_option0 = HAILO_DVM_OPTIONS_VDD_CORE;
    hailo_dvm_options_t dvm_option1 = HAILO_DVM_OPTIONS_VDD_TOP;
    hailo_dvm_options_t dvm_option2 = HAILO_DVM_OPTIONS_MIPI_AVDD;
    enum hailo_power_measurement_types_e measurement_type = HAILO_POWER_MEASUREMENT_TYPES__POWER;
    int measure_power = 0;
    hailo_activated_network_group activated_network_group = NULL;
    hailo_latency_measurement_result_t driver_result = {0};
    uint32_t latency_readout = 0;

    while ((opt = getopt(argc, argv, "b:f:i:c:n:pl")) != -1) {
        switch (opt) {
            case 'c': config = optarg; break;
            case 'i': iface =   optarg; break;
            case 'n': num_img = atoi(optarg); break;
            case 'b': nbatch = atoi(optarg); break;
            case 'f': source_fps = atoi(optarg); break;
            case 'l': write_log = 1; break;
            case 'p': measure_power = 1; break;
            case '?': fprintf (stderr, "Option -%c requires an argument.\n", optopt);
            default:
              fprintf(stderr, "Usage: %s -c HEF [-i INTERFACE] [-n NUM-IMAGES]\n\n", argv[0]);
              fprintf(stderr, "     -c HEF FILE       The HEF config file \n");
              fprintf(stderr, "     -i INTERFACE      The Interface, defaults to \'pcie\', can be also an Ethernet port name\n");
              fprintf(stderr, "     -n NUM-IMAGES     The number of images to process, defaults to 100\n");
              fprintf(stderr, "     -b BATCH          Batch size (def. 1)\n");
              fprintf(stderr, "     -f FPS            Emulate the source FPS (def. inf.)\n");
              fprintf(stderr, "     -l                Each receive thread will write a log file\n");
              fprintf(stderr, "     -p                Enable Power Measurement\n");
              exit(EXIT_FAILURE);
        }
    }

    if (num_img%nbatch != 0) {
        printf(BOLDRED);
        printf("-E- When specifying batch (%d), needs to make it a complete divisor of NUM-IMAGES (%d)\n", nbatch, num_img);
        printf(RESET);
        exit(1);
    }
    if (!strcmp(config,"")) {
        printf(BOLDRED);
        printf("-E- Must specify a path to existing HEF file\n");
        printf(RESET);
        exit(1);
    }
    printf(BOLDCYAN);
    printf("-I- Running on interface: %s for %d images\n", iface, num_img);
    printf(RESET);
    
    status = initiazlie_inference(&device, &input_stream, all_stream_infos, output_streams, &host_input_frame_size, 
            host_output_frame_size, config, &output_stream_cnt, &activated_network_group, iface, nbatch);
    REQUIRE_SUCCESS(status, l_release_device, "Failed to initialize inference");

    if (measure_power==1) {
        status = hailo_stop_power_measurement(device);
        status = hailo_set_power_measurement(device, 0, dvm_option0, measurement_type);
        status = hailo_set_power_measurement(device, 1, dvm_option1, measurement_type);
        status = hailo_set_power_measurement(device, 2, dvm_option2, measurement_type);
        status = hailo_start_power_measurement(device, DEFAULT_POWER_MEASUREMENT_DELAY_MS, HAILO_DEFAULT_INIT_AVERAGING_FACTOR, HAILO_DEFAULT_INIT_SAMPLING_PERIOD_US);
    }
    (void) clock_gettime(CLOCK_MONOTONIC, &start_time);    
    status = infer(input_stream, output_streams, all_stream_infos, output_stream_cnt, num_img, write_log, host_output_frame_size, host_input_frame_size, source_fps);
    
    REQUIRE_SUCCESS(status, l_release_device, "Inference failure");
    (void) clock_gettime(CLOCK_MONOTONIC, &end_time);

    if (measure_power==1)
        status = print_power_stats(device);

    // Getting the latency from the driver
    status = hailo_get_latency_measurement(activated_network_group, &driver_result);
    print_inference_stats(start_time, end_time, all_stream_infos, output_stream_cnt, num_img, latency_readout, write_log, driver_result, nbatch);

l_release_device:
    (void) hailo_deactivate_network_group(activated_network_group);
    if (device != NULL)
        (void) hailo_release_device(device);
    printf(MAGENTA);
    printf("-I- Finished gracefully\n");
    printf(RESET);
    return status;
}
