/**
 * Copyright 2020 (C) Hailo Technologies Ltd.
 * All rights reserved.
 *
 * Hailo Technologies Ltd. ("Hailo") disclaims any warranties, including, but not limited to,
 * the implied warranties of merchantability and fitness for a particular purpose.
 * This software is provided on an "AS IS" basis, and Hailo has no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 * You may use this software in the development of any project.
 * You shall not reproduce, modify or distribute this software without prior written permission.
 **/
/**
 * @ file example.c
 * This example demonstrates the basic data-path on HailoRT.
 * The program scans for Hailo-8 devices connected to a provided Ethernet interface, generates random dataset,
 * and runs it through the device.
 **/
#include <net/if.h>
#include <pthread.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <regex>
#include <dirent.h>
#include <linux/limits.h>
#include <time.h>
#include <queue>
#include <opencv2/opencv.hpp>
#include <opencv2/video/video.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <fstream>

#include <hailo/hailort.h>
#include "example_utils.hpp"
#include "ImageNetLabels.hpp"

#define NOF_STREAMS (5)
#define NSEC_IN_SEC (1e+9)
#define LATENCY_MEASUREMENTS (100)

using namespace std;
using namespace cv;

struct timespec sent_clock_t[LATENCY_MEASUREMENTS];
struct timespec recv_clock_t[NOF_STREAMS-1][LATENCY_MEASUREMENTS];

queue<Mat> input_image_queue;
pthread_mutex_t input_image_queue_m = PTHREAD_MUTEX_INITIALIZER;

queue< vector<uint8_t> > post_infer_queue[NOF_STREAMS-1];
pthread_mutex_t post_infer_queue_m[NOF_STREAMS-1] = PTHREAD_MUTEX_INITIALIZER;

string classification_post_process(vector<float32_t>& tensor, unsigned offset=0, float threshold=0.3) 
{
    int max_idx;
    static ImageNetLabels obj;
    max_idx = argmax(softmax(tensor));
    if (tensor[max_idx] < threshold) return "N\\A";
    return obj.imagenet_labelstring(max_idx);
}

cv::Mat _aspect_preserving_resize(const cv::Mat &image, int target_width)
{
    cv::Mat output;
    int min_dim = ( image.cols >= image.rows ) ? image.rows : image.cols;
    float scale = ( ( float ) target_width ) / min_dim;
    cv::resize( image, output, cv::Size(int(image.cols*scale), int(image.rows*scale)));
    return output;
}

cv::Mat _center_crop(cv::Mat& image, cv::Size& input_size) 
{
    cv::Rect myROI(int(image.cols/2-input_size.width/2), int(image.rows/2-input_size.height/2), input_size.width, input_size.height);
    cv::Mat croppedImage = image(myROI);
    return croppedImage;
}

cv::Mat classification_pre_process(const cv::Mat& image, cv::Size& input_size)
{
    cv::Mat output(input_size, CV_8UC3);
    int min_input_size = int(input_size.height * 1.14);
    output = _aspect_preserving_resize(image, min_input_size);
    return _center_crop(output, input_size);
}

bool source_is_npy(string& path) {
    const vector<string> suffixs = {"npy", "NPY"};
    for (string suffix : suffixs) {
        if (regex_search(path, regex(string(suffix) + "$")))
            return true;
    }
    return false;
}

void* _frame_lib_thread(void *args) {
    video_thread_args_t *v_args = (video_thread_args_t*)args;
    cv::Mat org_frame;
    vector<cv::String> file_names;
    cv::Mat pp_frame(v_args->input_stream_info->hw_shape.height, v_args->input_stream_info->hw_shape.height, CV_8UC3);
    unsigned int idx=0;
    cv::Size input_size;
    // std::ofstream outFile("tx_images.cpp.txt");

    cout << BOLDCYAN << "-I-----------------------------------------------" << endl
                     << "-I- [Frames-thread] Source: " << v_args->video_path << endl
                     << "-I- [Frames-thread] Output shape (" << v_args->input_stream_info->hw_shape.height << ", " << v_args->input_stream_info->hw_shape.height << ")" << endl
                     << "-I-----------------------------------------------" 
         << RESET    << endl;

    cv::glob(v_args->video_path, file_names, false);
    input_size.width = v_args->input_stream_info->hw_shape.width;
    input_size.height = v_args->input_stream_info->hw_shape.height;
    for (std::string file : file_names) {
        if (not(file.ends_with(".jpg") || file.ends_with(".png"))) {
            continue;
        }
        org_frame = cv::imread(file);
        if (org_frame.channels() == 3) {
            cv::cvtColor(org_frame, org_frame, cv::COLOR_BGR2RGB);
        }
        cout << "-I- [Frames-thread] Getting frame #"<< idx++ << ":" << file << endl;

        pp_frame = classification_pre_process(org_frame, input_size);
        // outFile << "TX Image "<<  idx << " " << file << endl;
        // outFile << pp_frame << endl;

        pthread_mutex_lock(&input_image_queue_m); 
        input_image_queue.push(pp_frame.clone());
        pthread_mutex_unlock(&input_image_queue_m); 
    }
    // Mark the final frame
    pp_frame.data[0] = 17;
    pp_frame.data[1] = 71;
    pp_frame.data[2] = 17;
    pp_frame.data[3] = 71;
    idx++;
    pthread_mutex_lock(&input_image_queue_m); 
    input_image_queue.push(pp_frame.clone());
    pthread_mutex_unlock(&input_image_queue_m); 
    cout << BOLDCYAN << "-I-----------------------------------------------" << endl
                     << "-I- [Frames-thread] FINISHED READING DIR - #" << idx-1 << endl
                     << "-I-----------------------------------------------" 
         << RESET    << endl;
    
    return NULL;
}

void* _video_source_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    video_thread_args_t *v_args = (video_thread_args_t*)args;
    Mat org_frame;
    Mat pp_frame(v_args->input_stream_info->hw_shape.height, v_args->input_stream_info->hw_shape.height, CV_8UC3);
    unsigned int idx=0;
    VideoCapture cap(v_args->video_path);

    if (!cap.isOpened()) {
        cout << "-E- Unable to open video stream" << endl;
        status = HAILO_INTERNAL_FAILURE;
    }
    Size shape = Size((int) cap.get(CV_CAP_PROP_FRAME_WIDTH), (int) cap.get(CV_CAP_PROP_FRAME_HEIGHT));

    cout << BOLDCYAN << "-I-----------------------------------------------" << endl;
    cout << "-I- [Video-thread] Source: " << v_args->video_path << endl;
    cout << "-I- [Video-thread] Input shape  (" << shape.width <<", "<<shape.height<<")" << endl;
    cout << "-I- [Video-thread] Output shape (" << v_args->input_stream_info->hw_shape.height << ", " << v_args->input_stream_info->hw_shape.height << ")" << endl;
    cout << "-I-----------------------------------------------" << RESET << endl;

    while (true && status==HAILO_SUCCESS) {
        if (++idx%500==0) 
            cout << "-I- [Video-thread] Getting frame: "<< idx << endl;
        cap >> org_frame;
        if (org_frame.empty())
            break;
        
        resize(org_frame, pp_frame, pp_frame.size());

        pthread_mutex_lock(&input_image_queue_m); 
        input_image_queue.push(pp_frame.clone());
        pthread_mutex_unlock(&input_image_queue_m); 
    }
    // Mark the final frame
    pp_frame.data[0] = 17;
    pp_frame.data[1] = 71;
    pp_frame.data[2] = 17;
    pp_frame.data[3] = 71;
    idx++;
    pthread_mutex_lock(&input_image_queue_m); 
    input_image_queue.push(pp_frame);
    pthread_mutex_unlock(&input_image_queue_m); 
    cout << BOLDCYAN << "-I-----------------------------------------------" << endl;
    cout << "-I- [Video-thread] FINISHED MOVIE STREAM - " << idx << endl;
    cout << "-I-----------------------------------------------" << RESET << endl;
    cap.release();

    return (void *)status;
}

void* _send_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    write_thread_args_t *write_args = (write_thread_args_t*)args;
    unsigned lat_counter = 0;
    struct timespec ts;
    Mat curr_img(write_args->input_stream_info->shape.height, write_args->input_stream_info->shape.width, CV_8UC3);
    int counter = 0;
    vector<uint8_t> array;
    
    array.resize(write_args->input_shape_size);

    while (true) {
        if (input_image_queue.empty()) {
            cout << "-I- Waiting for the image queue to fill" << endl;
            sleep(1);
            continue;
        }
        pthread_mutex_lock(&input_image_queue_m);
        curr_img = input_image_queue.front();
        input_image_queue.pop();
        pthread_mutex_unlock(&input_image_queue_m);
        if (counter>10 && (lat_counter < LATENCY_MEASUREMENTS)) {
            clock_gettime(CLOCK_REALTIME, &ts);
            sent_clock_t[lat_counter++] = ts;
        }
        if (counter % 500==0) {
            clock_gettime(CLOCK_REALTIME, &ts);
            printf("-I- [%10ld.%ld s] Send frame [%3d] 0x%x%x\n", (long)ts.tv_sec, ts.tv_nsec/1000000, counter, curr_img.data[0], curr_img.data[1]);
        }
        counter++;
        if (curr_img.data[0] == 17 && curr_img.data[1] == 71 && curr_img.data[2] == 17 && curr_img.data[3] == 71) {
            cout << "-I- [TX-Thread] Received stop pattern - " << counter << endl;
            sleep (1);
            break;
        }

        if (curr_img.isContinuous()) {
            int totalsz = curr_img.dataend-curr_img.datastart;
            array.assign(curr_img.datastart, curr_img.datastart + totalsz);
        } else {
            int rowsz = CV_ELEM_SIZE(curr_img.type()) * curr_img.cols;
            for (int i = 0; i < curr_img.rows; ++i) {
                array.insert(array.end(), curr_img.ptr<uint8_t>(i), curr_img.ptr<uint8_t>(i) + rowsz);
            }
        }
        // status = hailo_stream_sync_write_all_raw_buffer(write_args->input_stream, &src_data, 0, write_args->input_shape_size * sizeof(float32_t));
        status = hailo_stream_sync_write_all_raw_buffer(write_args->input_stream, array.data(), 0, write_args->input_shape_size * sizeof(uint8_t));
        if (status != HAILO_SUCCESS) {
            printf("-E- hailo_stream_sync_write_all_raw_buffer failed");
            break;
        }
    }
    write_args->status = status;
    write_args->num_images = counter;

    return NULL;
}

void* _recv_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    recv_thread_args_t *recv_args = (recv_thread_args_t *)args;
    struct timespec ts;
    ofstream outFile;
    int lat_counter = 0;
    int counter = 1;
    string detection;
    std::vector<uint8_t> recv_array;

    printf("-I- Recv thread %d started\n", recv_args->tid);
    if (recv_args->write_log==1) {
        char *log_name = (char*)malloc(20 * sizeof(char));
        sprintf(log_name, "tid_%d.log", recv_args->tid);
        outFile = ofstream("rx_images.cpp.txt");
        free(log_name);
    }

    recv_array.resize(recv_args->output_shape_size);
    while (true) {
        status = hailo_stream_sync_read_all_raw_buffer(recv_args->output_stream, recv_array.data(), 0, recv_args->output_shape_size * sizeof(uint8_t));
        if (status != HAILO_SUCCESS) {
            cout << "-E- hailo_stream_sync_read_all_raw_buffer failed" << endl;
            break;
        }
    
        clock_gettime(CLOCK_REALTIME, &ts);
        std::vector<float32_t> float32_tVec(recv_array.begin(), recv_array.end());
        detection = classification_post_process(float32_tVec, 1);
        printf("-I- [%10ld.%ld s] TID:%d Recv %3d: %s\n",  (long)ts.tv_sec, ts.tv_nsec/1000000, recv_args->tid, counter, detection.c_str());

        if (counter>10 && lat_counter < LATENCY_MEASUREMENTS) {
            clock_gettime(CLOCK_REALTIME, &ts);
            recv_clock_t[recv_args->tid][lat_counter++] = ts;
            recv_args->lat_counter = lat_counter;
        }
        counter++;
        if (recv_args->write_log==1) {
            outFile << "RX " << counter << endl;
            for (const auto &val : recv_array) outFile << val << " ";
            outFile << endl;
        }
    }
    
    recv_args->status = status;
    return (void *)status;
}

bool source_is_video(string& path) {
    const vector<string> suffixs = {"mkv", "avi", "webm"};
    for (string suffix : suffixs) {
        if (regex_search(path, regex(string(suffix) + "$")))
            return true;
    }
    return false;
}

hailo_status infer(
    hailo_input_stream input_stream, hailo_stream_info_t *all_stream_info,
    hailo_output_stream *output_streams, int output_stream_cnt, int* num_imgs, int write_log,
    string video_path, int* lat_counter, size_t input_size, size_t* output_size)
{
    hailo_status status = HAILO_SUCCESS;
    pthread_t write_thread = 1;
    recv_thread_args_t recv_args[NOF_STREAMS-1];
    video_thread_args_t video_args;
    pthread_t recv_threads[NOF_STREAMS-1];
    pthread_t video_thread = 2;
    write_thread_args_t write_args;

    write_args.input_stream_info = &all_stream_info[0];
    write_args.input_stream = input_stream;
    write_args.status = HAILO_SUCCESS;
    write_args.output_streams_cnt = output_stream_cnt;
    write_args.input_shape_size = input_size;
    write_args.num_images = 0;
    
    video_args.input_stream_info = &all_stream_info[0];
    video_args.video_path = video_path.c_str();
    for (int ii=0; ii<output_stream_cnt; ii++) {
        recv_args[ii].output_stream_info = &all_stream_info[ii+1];
        recv_args[ii].output_stream = output_streams[ii];
        recv_args[ii].tid = ii;
        recv_args[ii].status = HAILO_SUCCESS;
        recv_args[ii].lat_counter = 0;
        recv_args[ii].write_log = write_log;
        recv_args[ii].output_shape_size = output_size[ii];
        (void) pthread_create(&recv_threads[ii], NULL, _recv_thread, &recv_args[ii]);
    };
    
    if (source_is_video(video_path))
        (void) pthread_create(&video_thread, NULL, _video_source_thread, &video_args);
    else
        (void) pthread_create(&video_thread, NULL, _frame_lib_thread, &video_args);
    (void) pthread_create(&write_thread, NULL, _send_thread, &write_args);
    
    pthread_join(video_thread, NULL);
    pthread_join(write_thread, NULL);
    
    if (HAILO_SUCCESS != write_args.status) {
        printf("-E- write_thread failed\n");
        status = HAILO_INTERNAL_FAILURE;
    }
    for (int ii=0; ii<output_stream_cnt; ii++) {
        pthread_cancel(recv_threads[ii]);
        printf("-I- Sending SIGSTOP to Recv thread #%d status=%d\n", recv_args[ii].tid, recv_args[ii].status);
        if (HAILO_SUCCESS != recv_args[ii].status) {
            printf("-E- Recv thread %d failed\n", recv_args[ii].tid);
            status = HAILO_INTERNAL_FAILURE;
        }
    }
    *num_imgs = write_args.num_images;
    *lat_counter = recv_args[0].lat_counter;
    return status;
}

double get_time_from_ts(struct timespec ts) {
    double result = (double)(ts.tv_sec * 1000 + (ts.tv_nsec / 1000000));
    return result;
}

double calc_latency(int count, int lat_counter) {
    double result = 0;
    double cur_rcv;
    double cur_snd;

    for (int j=0; j<LATENCY_MEASUREMENTS; j++) {
        cur_snd = get_time_from_ts(sent_clock_t[j]);
        cur_rcv = get_time_from_ts(recv_clock_t[0][j]);
        for (int ii=1; ii<count; ii++) {
            if (get_time_from_ts(recv_clock_t[ii][j]) > cur_rcv) {
                cur_rcv = get_time_from_ts(recv_clock_t[ii][j]);
            }
        }
        result += (cur_rcv - cur_snd);
    }
    return result/lat_counter;
}

void print_inference_stats(struct timespec start_time, struct timespec end_time, hailo_stream_info_t *all_stream_infos, int output_stream_cnt, 
    int num_images, int lat_counter, size_t input_size, size_t* output_size) {
    double start_time_secs = (double)start_time.tv_sec + ((double)start_time.tv_nsec / NSEC_IN_SEC);
    double end_time_secs = (double)end_time.tv_sec + ((double)end_time.tv_nsec / NSEC_IN_SEC);
    double infer_time_secs = end_time_secs - start_time_secs;
    static float mbit_per_byte = 8.0f / 1024.0f / 1024.0f;
    uint32_t send_frame_size = input_size;
    uint32_t recv_frame_size = 0;

    printf(BOLDGREEN);
    printf("-I-----------------------------------------------\n");
    printf("-I- Total time:      %4.2lf sec\n", infer_time_secs);
    printf("-I- Average FPS:     %4.2lf\n", num_images / infer_time_secs);
    printf("-I- Average Latency: %3.2lf ms\n", calc_latency(output_stream_cnt, lat_counter));
    printf("-I- Send data rate:  %-4.2lf Mbit/s\n",
        (double)(num_images) * send_frame_size * mbit_per_byte / infer_time_secs);
    for (int i=1; i<=output_stream_cnt; i++) {
        recv_frame_size = output_size[i]; //hw_frame_size;
        printf("-I- Recv[%d] data rate: %-4.2lf Mbit/s\n", i,
            (double)(num_images) * recv_frame_size * mbit_per_byte / infer_time_secs);
        printf("-I-----------------------------------------------\n");
    }
    printf(RESET);
}

const char** get_jlf_files_form_path(const char *dir_name, uint8_t *actual_number_of_jlfs_files) {
    static char jlf_files[HAILO_MAX_NUMBER_OF_JLFS][PATH_MAX];
    static const char *res_jlf_files[HAILO_MAX_NUMBER_OF_JLFS];
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    uint8_t i = 0;

    dir = opendir(dir_name);
    if (NULL == dir) {
        return NULL;
    }

    entry = readdir(dir);
    while (NULL != entry) {
        if (entry->d_name[0] != '.') {
            (void)snprintf(jlf_files[i], sizeof(jlf_files[i]), "%s%s", dir_name, entry->d_name);
            res_jlf_files[i] = jlf_files[i];
            i++;
        }
        entry = readdir(dir);
    }
    (void) closedir(dir);
    *actual_number_of_jlfs_files = i;
    return res_jlf_files;
}

const char* get_direction_name(hailo_stream_direction_t dir) {
    switch (dir) {
        case HAILO_H2D_STREAM: return "Input";
        case HAILO_D2H_STREAM: return "Output";
        case HAILO_STREAM_DIRECTION_MAX_ENUM: return "Wrong";
    }
    return "Wrong";
}

void print_net_banner(hailo_stream_info_t *all_stream_infos, int cnt) {
    printf(BOLDCYAN);
    printf("-I-----------------------------------------------\n");
    for (int ii=0; ii<cnt; ii++) {
        printf("-I- %s[%d]: %s (%d, %d, %d) index:%d\n", get_direction_name(all_stream_infos[ii].direction), ii, all_stream_infos[ii].name, all_stream_infos[ii].shape.height, all_stream_infos[ii].shape.width, 
            all_stream_infos[ii].shape.features, all_stream_infos[0].index);
    }
    printf("-I-----------------------------------------------\n");
    printf(RESET);
}

hailo_status print_debug_stats(hailo_device device) {
    uint32_t address;
    uint8_t* data = (uint8_t*)malloc(4);
    uint32_t size = 4;
    hailo_status status = HAILO_SUCCESS;

    // rx_jabbers
    address = 0x0010918C;
    status = hailo_read_memory(device, address, data, size);
    printf("\033[1;31m");
    printf("-I-----------------------------------------------\n");
    printf("-D- RX_JABBERS: 0x%d\n", *data);

    // fcs_errors
    address = 0x00109190;
    status = hailo_read_memory(device, address, data, size);
    printf("-D- FCS_ERRORS: 0x%d\n", *data);
    printf("-I-----------------------------------------------\n");
    printf(RESET);

    return status;
}

hailo_status activate_input_stream(hailo_stream_info_t info, hailo_input_stream& input_stream, hailo_device device, hailo_jlf jlf, string iface, size_t* input_size) {
    hailo_status status = HAILO_SUCCESS;
    hailo_eth_input_stream_params_t input_stream_params = HAILO_ETH_INPUT_STREAM_PARAMS_DEFAULT;
    hailo_pcie_input_stream_params_t pcie_input_stream_params = HAILO_PCIE_STREAM_PARAMS_DEFAULT;

    if (info.format.order == HAILO_FORMAT_ORDER_NC) {
        input_stream_params.base_params.user_buffer_format.order = info.format.order;
        pcie_input_stream_params.base_params.user_buffer_format.order = info.format.order;
    }

    cout << "-I- Setting input format.order to: " << input_stream_params.base_params.user_buffer_format.order << endl;
    
    if (iface.compare("pcie") == 0) {
        status = hailo_create_pcie_input_stream_from_jlf_by_index(device, jlf, info.index, &pcie_input_stream_params, &input_stream);
        *input_size = hailo_get_host_frame_size(&info, &pcie_input_stream_params.base_params);
    } else {
        status = hailo_create_eth_input_stream_from_jlf_by_index(device, jlf, info.index, &input_stream_params, &input_stream);
        *input_size = hailo_get_host_frame_size(&info, &input_stream_params.base_params);
    }
    status = hailo_activate_input_stream(device, input_stream);
    return status;
}

hailo_status activate_output_stream(hailo_stream_info_t info, hailo_output_stream& output_stream, hailo_device device, hailo_jlf jlf, string iface, size_t* output_size) {
    hailo_status status = HAILO_SUCCESS;
    hailo_eth_output_stream_params_t output_stream_params = HAILO_ETH_OUTPUT_STREAM_PARAMS_DEFAULT;
    hailo_pcie_output_stream_params_t pcie_output_stream_params = HAILO_PCIE_STREAM_PARAMS_DEFAULT;

    if (info.format.order == HAILO_FORMAT_ORDER_NC ||
        info.format.order == HAILO_FORMAT_ORDER_NHW ||
        info.format.order == HAILO_FORMAT_ORDER_HAILO_NMS) {
            output_stream_params.base_params.user_buffer_format.order = info.format.order;
            pcie_output_stream_params.base_params.user_buffer_format.order = info.format.order;
    }

    cout << "-I- Setting output format.order to: " << output_stream_params.base_params.user_buffer_format.order << endl;

    if (iface.compare("pcie") == 0) {
        status = hailo_create_pcie_output_stream_from_jlf_by_index(device, jlf, info.index, &pcie_output_stream_params, &output_stream);    
        *output_size = hailo_get_host_frame_size(&info, &pcie_output_stream_params.base_params);
    } else {
        status = hailo_create_eth_output_stream_from_jlf_by_index(device, jlf, info.index, &output_stream_params, &output_stream);
        *output_size = hailo_get_host_frame_size(&info, &output_stream_params.base_params);
    }
    status = hailo_activate_output_stream(device, output_stream);
    
    return status;
}

int main(int argc, char **argv) {
    hailo_status status = HAILO_UNINITIALIZED;
    size_t number_of_devices = 0;
    hailo_eth_device_info_t device_info = {0};
    hailo_pcie_device_info_t pcie_device_info = {0};
    hailo_device device = NULL;
    hailo_jlf jlf = NULL;
    uint8_t jlf_buffer[48*1024];
    uint8_t actual_number_of_jlfs_files = 0;
    hailo_input_stream input_stream = NULL;
    hailo_output_stream output_streams[NOF_STREAMS];
    hailo_stream_info_t all_stream_infos[NOF_STREAMS];
    size_t host_input_frame_size = 0;
    size_t host_output_frame_size[NOF_STREAMS];
    const char **jlf_files = NULL;
    struct timespec start_time = {0};
    struct timespec end_time = {0};
    size_t number_of_streams = 0;
    int output_stream_cnt = 0;
    string jlf_dir = "./JLFs/";
    int opt;
    string iface = "pcie";
    string video;
    int lat_counter = 0;
    int num_img = 0;
    int debug = 0;
    int write_log = 0;

    while ((opt = getopt(argc, argv, "i:j:v:dl")) != -1) {
        switch (opt) {
            case 'j': jlf_dir = optarg; break;
            case 'i': iface =   optarg; break;
            case 'd': debug = 1; break;
            case 'v': video = optarg; break;
            case 'l': write_log = 1; break;
            case '?': fprintf (stderr, "Option -%c requires an argument.\n", optopt);
            default:
              fprintf(stderr, "Usage: %s -i INTERFACE -j JLF-DIR -v Path/to/video\n\n", argv[0]);
              fprintf(stderr, "     -i INTERFACE      The device interface, defaults to \'pcie\', for Ethernet put the host interface\n");
              fprintf(stderr, "     -j JLF-DIR        The JLFs directory, defaults to \'./JLFs/\'\n");
              fprintf(stderr, "     -v Path/to/Video  The path to the video file\n");
              fprintf(stderr, "     -d                Read and print debug registers from FW\n");
              fprintf(stderr, "     -l                Each receive thread will write a log file\n");
              exit(EXIT_FAILURE);
        }
    }
    printf(BOLDCYAN);
    printf("-I- Running on interface: %s \n", iface.c_str());
    printf("-I- Reading JLFs from: %s\n", jlf_dir.c_str());
    printf(RESET);
    if (iface.compare("pcie") == 0) {
        status = hailo_scan_pcie_devices(&pcie_device_info, 1, &number_of_devices);
    } else {
        status = hailo_scan_ethernet_devices((char *)iface.c_str(), &device_info, 1, &number_of_devices, HAILO_DEFAULT_ETH_SCAN_TIMEOUT_MS);
    }
    if (0 == number_of_devices || status != HAILO_SUCCESS) {
        printf("-E- No device found on the given interface\n");
        return status;
    }
    if (iface.compare("pcie") == 0) {
        status = hailo_create_pcie_device(&pcie_device_info, &device);
    } else {
        status = hailo_create_ethernet_device(&device_info, &device);
    }
    if (status != HAILO_SUCCESS) {
        printf("-E- Failed to create eth_device\n");
        return status;
    }
    jlf_files = get_jlf_files_form_path((char *)jlf_dir.c_str(), &actual_number_of_jlfs_files);
    if (NULL == jlf_files) {
        printf("-E- Failed to get jlf files from path\n");
        (void) hailo_release_device(device);
        return status;
    }

    status = hailo_create_jlf_files(jlf_files, actual_number_of_jlfs_files, jlf_buffer, sizeof(jlf_buffer), &jlf);
    if (status != HAILO_SUCCESS) {
        printf("-E- Failed to create jlf files\n");
        (void) hailo_release_device(device);
        return status;
    }
    status = hailo_configure_device_from_jlf(device, jlf, jlf_buffer, sizeof(jlf_buffer));
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to configure device from jlf");
    
    // NEDEN
    status = hailo_jlf_get_all_stream_infos(jlf, all_stream_infos, NOF_STREAMS, &number_of_streams);
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to get all stream info");

    for (size_t i=0;i<number_of_streams;i++) {
        if (all_stream_infos[i].direction==HAILO_H2D_STREAM) {
            activate_input_stream(all_stream_infos[i], input_stream, device, jlf, iface, &host_input_frame_size);
        } else {
            activate_output_stream(all_stream_infos[i], output_streams[output_stream_cnt], device, jlf, iface, &host_output_frame_size[output_stream_cnt]);
            output_stream_cnt++;
        }
    }
    if (status != HAILO_SUCCESS) {
        printf("-E- Failed to activate streams\n");
        return status;
    }
    print_net_banner(all_stream_infos, number_of_streams);

    (void) clock_gettime(CLOCK_MONOTONIC, &start_time);

    status = infer(input_stream, all_stream_infos, output_streams, output_stream_cnt, &num_img, write_log, video, &lat_counter, host_input_frame_size, host_output_frame_size);
    REQUIRE_SUCCESS(status, l_release_output_stream, "Inference failure");

    (void) clock_gettime(CLOCK_MONOTONIC, &end_time);

    print_inference_stats(start_time, end_time, all_stream_infos, output_stream_cnt, num_img, lat_counter, host_input_frame_size, host_output_frame_size);
    status = HAILO_SUCCESS;

l_release_output_stream:
    for (int i=0;i<output_stream_cnt;i++) {
        (void) hailo_release_output_stream(device, output_streams[i]);
    }
    if (debug==1) {
        print_debug_stats(device);
        REQUIRE_SUCCESS(status, l_release_input_stream, "Failed to read debug registers");
    }
l_release_input_stream:
    (void) hailo_release_input_stream(device, input_stream);
l_release_jlf:
    (void) hailo_release_jlf(jlf);

    (void) hailo_release_device(device);
    return status;
}
