/**
 * Copyright 2020 (C) Hailo Technologies Ltd.
 * All rights reserved.
 *
 * Hailo Technologies Ltd. ("Hailo") disclaims any warranties, including, but not limited to,
 * the implied warranties of merchantability and fitness for a particular purpose.
 * This software is provided on an "AS IS" basis, and Hailo has no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 * You may use this software in the development of any project.
 * You shall not reproduce, modify or distribute this software without prior written permission.
 **/
/**
 * @ file example.c
 * This example demonstrates the basic data-path on HailoRT.
 * The program scans for Hailo-8 devices connected to a provided Ethernet interface, generates random dataset,
 * and runs it through the device.
 **/

#include <net/if.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <linux/limits.h>
#include <time.h>

#include <hailo/hailort.h>

#define INFER_FRAME_COUNT (5000)
#define UNIQUE_FRAMES_COUNT (3)
#define NSEC_IN_SEC (1e+9)
#define JLF_DIR "shortcut_50_2928_1/"

#define FREE(var)                           \
    do {                                    \
        if (NULL != (var)) {                \
            free(var);                      \
            var = NULL;                     \
        }                                   \
    } while(0)

#define REQUIRE_SUCCESS(status, label, ...) \
    do {                                    \
        if (HAILO_SUCCESS != (status)) {    \
            printf(__VA_ARGS__);            \
            printf("\n");                   \
            goto label;                     \
        }                                   \
    } while(0)

typedef struct write_thread_args_t {
    uint8_t *src_data;
    size_t host_frame_size;
    hailo_input_stream input_stream;
    hailo_status status;
} write_thread_args_t;

void* write_thread_func(void *args) 
{
    hailo_status status = HAILO_UNINITIALIZED;
    write_thread_args_t *write_args = (write_thread_args_t*)args;

    for (uint32_t i = 0; i < INFER_FRAME_COUNT; i++) {
        status = hailo_stream_sync_write_all_raw_buffer(write_args->input_stream,
            (uint8_t*)write_args->src_data + (i % UNIQUE_FRAMES_COUNT) * write_args->host_frame_size,
            0, write_args->host_frame_size);
        REQUIRE_SUCCESS(status, l_exit, "hailo_stream_sync_write_all_raw_buffer failed");
    }

    status = HAILO_SUCCESS;
l_exit:
    write_args->status = status;
    return NULL;
}

hailo_status read_and_compare(
    hailo_output_stream output_stream, size_t host_output_frame_size, uint8_t *dst_data,
    size_t host_input_frame_size, uint8_t *src_data)
{
    hailo_status status = HAILO_UNINITIALIZED;
    int cmp_result = -1;

    for (uint32_t i = 0; i < INFER_FRAME_COUNT; i++) {
        status = hailo_stream_sync_read_all_raw_buffer(output_stream, dst_data, 0, host_output_frame_size);
        REQUIRE_SUCCESS(status, l_exit, "hailo_stream_sync_read_all_raw_buffer failed");

        cmp_result = memcmp(src_data + (i % UNIQUE_FRAMES_COUNT) * host_input_frame_size, dst_data,
            host_input_frame_size);
        if (0 != cmp_result) {
            printf("Diff in frame number %d\n", i);
            // Still success
        }
    }

l_exit:
    return status;
}

hailo_status infer(
    hailo_input_stream input_stream, size_t host_input_frame_size, uint8_t *src_data,
    hailo_output_stream output_stream, size_t host_output_frame_size, uint8_t *dst_data)
{
    hailo_status status = HAILO_UNINITIALIZED;
    pthread_t write_thread = 1;

    write_thread_args_t write_args = {
        .src_data = src_data,
        .host_frame_size = host_input_frame_size,
        .input_stream = input_stream,
        .status = HAILO_UNINITIALIZED
    };

    // Run write
    (void) pthread_create(&write_thread, NULL, write_thread_func, &write_args);
    // Receive data in main thread and compare results

    status = read_and_compare(output_stream, host_output_frame_size, dst_data, host_input_frame_size, src_data);
    if (HAILO_SUCCESS != status) {
        printf("read_and_compare failed\n");
        status = HAILO_INTERNAL_FAILURE;
        // Keep waiting for write thread
    }

    pthread_join(write_thread, NULL);
    if (HAILO_SUCCESS != write_args.status) {
        printf("write_thread failed");
        status = HAILO_INTERNAL_FAILURE;
    }

    return status;
}

void print_inference_stats(struct timespec start_time, struct timespec end_time, uint32_t send_frame_size,
    uint32_t recv_frame_size)
{
    double start_time_secs = (double)start_time.tv_sec + ((double)start_time.tv_nsec / NSEC_IN_SEC);
    double end_time_secs = (double)end_time.tv_sec + ((double)end_time.tv_nsec / NSEC_IN_SEC);
    double infer_time_secs = end_time_secs - start_time_secs;
    static float mbit_per_byte = 8.0f / 1000.0f / 1000.0f;
    printf("Inference time: %lf seconds\n", infer_time_secs);
    printf("Inference fps: %lf\n", INFER_FRAME_COUNT / infer_time_secs);
    printf("Inference send data rate: %lf Mbit/s\n",
        (double)(INFER_FRAME_COUNT) * send_frame_size * mbit_per_byte / infer_time_secs);
    printf("Inference recv data rate: %lf Mbit/s\n",
        (double)(INFER_FRAME_COUNT) * recv_frame_size * mbit_per_byte / infer_time_secs);
}

const char** get_jlf_files_form_path(const char *dir_name, uint8_t *actual_number_of_jlfs_files)
{
    static char jlf_files[HAILO_MAX_NUMBER_OF_JLFS][PATH_MAX];
    static const char *res_jlf_files[HAILO_MAX_NUMBER_OF_JLFS];
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    uint8_t i = 0;

    dir = opendir(dir_name);
    if (NULL == dir) {
        return NULL;
    }

    entry = readdir(dir);
    while (NULL != entry) {
        if (entry->d_name[0] != '.') {
            (void)snprintf(jlf_files[i], sizeof(jlf_files[i]), "%s%s", dir_name, entry->d_name);
            res_jlf_files[i] = jlf_files[i];
            i++;
        }
        entry = readdir(dir);
    }
    (void) closedir(dir);
    *actual_number_of_jlfs_files = i;
    return res_jlf_files;
}

int main(int argc, char **argv)
{
    hailo_status status = HAILO_UNINITIALIZED;
    size_t number_of_devices = 0;
    hailo_eth_device_info_t device_info = {0};
    hailo_device device = NULL;
    hailo_jlf jlf = NULL;
    uint8_t jlf_buffer[16*1024];
    uint8_t actual_number_of_jlfs_files = 0;
    hailo_input_stream input_stream = NULL;
    hailo_output_stream output_stream = NULL;
    hailo_eth_input_stream_params_t input_stream_params = HAILO_ETH_INPUT_STREAM_PARAMS_DEFAULT;
    hailo_eth_output_stream_params_t output_stream_params = HAILO_ETH_OUTPUT_STREAM_PARAMS_DEFAULT;
    hailo_stream_info_t input_stream_info = {0};
    hailo_stream_info_t output_stream_info = {0};
    const char **jlf_files = NULL;
    struct timespec start_time = {0};
    struct timespec end_time = {0};
    uint8_t *src_data = NULL;
    uint8_t *dst_data = NULL;

    if (2 != argc) {
        printf("Wrong amount of arguments was provided (excpecting: 1, provided: %i)\n", (argc - 1));
        goto l_exit;
    }

    status = hailo_scan_ethernet_devices(argv[1], &device_info, 1, &number_of_devices,
        HAILO_DEFAULT_ETH_SCAN_TIMEOUT_MS);
    REQUIRE_SUCCESS(status, l_exit, "Failed to scan for eth_devices");
    if (0 == number_of_devices) {
        printf("No device found on the given interface\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_exit;
    }

    status = hailo_create_ethernet_device(&device_info, &device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create eth_device");

    jlf_files = get_jlf_files_form_path(JLF_DIR, &actual_number_of_jlfs_files);
    if (NULL == jlf_files) {
        printf("Failed to get jlf files from path\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_release_device;
    }

    status = hailo_create_jlf_files(jlf_files, actual_number_of_jlfs_files, jlf_buffer, sizeof(jlf_buffer), &jlf);
    REQUIRE_SUCCESS(status, l_release_device, "Failed to create jlf files");

    status = hailo_configure_device_from_jlf(device, jlf, jlf_buffer, sizeof(jlf_buffer));
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to configure device from jlf");

    // Create and activate streams
    status = hailo_create_eth_input_stream_from_jlf_by_index(device, jlf, 0, &input_stream_params, &input_stream);
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to create eth_input_stream");

    status = hailo_create_eth_output_stream_from_jlf_by_index(device, jlf, 1, &output_stream_params, &output_stream);
    REQUIRE_SUCCESS(status, l_release_input_stream, "Failed to create eth_output_stream");

    status = hailo_jlf_get_stream_info_by_index(jlf, 0, HAILO_H2D_STREAM, &input_stream_info);
    REQUIRE_SUCCESS(status, l_release_output_stream, "Failed to get eth_input_stream");

    status = hailo_jlf_get_stream_info_by_index(jlf, 1, HAILO_D2H_STREAM, &output_stream_info);
    REQUIRE_SUCCESS(status, l_release_output_stream, "Failed to get eth_output_stream");

    status = hailo_activate_input_stream(device, input_stream);
    REQUIRE_SUCCESS(status, l_release_output_stream, "Failed to activate input stream");

    status = hailo_activate_output_stream(device, output_stream);
    REQUIRE_SUCCESS(status, l_release_output_stream, "Failed to activate output stream");

    size_t host_input_frame_size = hailo_get_host_frame_size(&input_stream_info, &input_stream_params.base_params);
    size_t host_output_frame_size = hailo_get_host_frame_size(&output_stream_info, &output_stream_params.base_params);
    src_data = (uint8_t*)malloc(host_input_frame_size * UNIQUE_FRAMES_COUNT);
    dst_data = (uint8_t*)malloc(host_output_frame_size);
    if ((NULL == src_data) || (NULL == dst_data)) {
        printf("Failed to allocate buffers\n");
        status = HAILO_OUT_OF_HOST_MEMORY;
        goto l_release_buffers;
    }
    for(size_t i = 0; i < host_input_frame_size * UNIQUE_FRAMES_COUNT; i++) {
        src_data[i] = (uint8_t)(rand() % 256);
    }

    // Run inference and compare results
    (void) clock_gettime(CLOCK_MONOTONIC, &start_time);
    status = infer(
        input_stream, host_input_frame_size, src_data,
        output_stream, host_output_frame_size, dst_data);
    (void) clock_gettime(CLOCK_MONOTONIC, &end_time);
    REQUIRE_SUCCESS(status, l_release_buffers, "Inference failure");

    print_inference_stats(start_time, end_time, input_stream_info.hw_frame_size,
        output_stream_info.hw_frame_size);

    status = HAILO_SUCCESS;
l_release_buffers:
    FREE(dst_data);
    FREE(src_data);
l_release_output_stream:
    (void) hailo_release_output_stream(device, output_stream);
l_release_input_stream:
    (void) hailo_release_input_stream(device, input_stream);
l_release_jlf:
    (void) hailo_release_jlf(jlf);
l_release_device:
    (void) hailo_release_device(device);
l_exit:
    return status;
}
