/**
 * Copyright 2020 (C) Hailo Technologies Ltd.
 * All rights reserved.
 *
 * Hailo Technologies Ltd. ("Hailo") disclaims any warranties, including, but not limited to,
 * the implied warranties of merchantability and fitness for a particular purpose.
 * This software is provided on an "AS IS" basis, and Hailo has no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 * You may use this software in the development of any project.
 * You shall not reproduce, modify or distribute this software without prior written permission.
 **/
/**
 * @ file example.c
 * This example demonstrates the basic data-path on HailoRT.
 * The program scans for Hailo-8 devices connected to a provided Ethernet interface, generates random dataset,
 * and runs it through the device.
 **/
#include <net/if.h>
#include <pthread.h>
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <linux/limits.h>
#include <time.h>

#include <hailo/hailort.h>
#include "example_utils.h"

#define NOF_STREAMS (15)
#define NSEC_IN_SEC (1e+9)
#define LATENCY_MEASUREMENTS (100)

struct timespec sent_clock_t[LATENCY_MEASUREMENTS];
struct timespec recv_clock_t[NOF_STREAMS-1][LATENCY_MEASUREMENTS];
unsigned int actual_measurments;

const char* get_direction_name(hailo_stream_direction_t dir) {
    switch (dir) {
        case HAILO_H2D_STREAM: return "Input";
        case HAILO_D2H_STREAM: return "Output";
        case HAILO_STREAM_DIRECTION_MAX_ENUM: return "Wrong";
    }
    return "Wrong";
}

void* _send_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    write_thread_args_t *write_args = (write_thread_args_t*)args;
    uint32_t datasize = write_args->host_frame_size;
    uint8_t* src_data = (uint8_t*)malloc(datasize);
    unsigned lat_counter = 0;
    uint32_t flag_100 = 0;
    struct timespec ts;
    FILE *fp;
    float delay = 0.0;

    if (NULL == src_data) {
        printf("-E- Failed to allocate buffers\n");
        status = HAILO_OUT_OF_HOST_MEMORY;
        goto l_release_buffers;
    }
    for(size_t i = 0; i < datasize; i++) {
        src_data[i] = (uint8_t)(rand() % 256);
    }
    printf("-I- Input datasize: %d\n", datasize);
    if (write_args->write_log==1) {
        char *log_name = (char*)malloc(20 * sizeof(char));
        sprintf(log_name, "tx_tid_%d.log", write_args->tid);
        fp = fopen(log_name, "w");
        free(log_name);
    }
    flag_100 = (uint32_t)write_args->num_images / 100;
    if (flag_100==0)
        flag_100 = 1;

    if (write_args->source_fps > 0) {
        // Adding 20% to match the requested FPS
        delay = 800000.0 / write_args->source_fps;
        printf(CYAN);
        printf("-I- Setting the delay to %f usec\n", delay);
        printf(RESET);
    }
    for (uint32_t i = 1; i <= (uint32_t)write_args->num_images; i++) {
        if (delay > 0.0)
            usleep(delay);
        if ((i % flag_100==0) && (lat_counter < LATENCY_MEASUREMENTS)) {
            clock_gettime(CLOCK_REALTIME, &ts);
            printf("-I- [%10ld.%ld s] Send frame [%3d/%3d] 0x%x%x%x%x\n", (long)ts.tv_sec, ts.tv_nsec/1000000, i, write_args->num_images, src_data[0], src_data[1], src_data[2], src_data[3]);
            sent_clock_t[lat_counter++] = ts;
        }
        if (write_args->write_log==1)
            fprintf(fp, "%d Sending [%10ld.%ld s] %3d/%3d %x\n", write_args->tid, (long)ts.tv_sec, ts.tv_nsec/1000000, i, write_args->num_images, *src_data);
        status = hailo_stream_sync_write_all_raw_buffer(write_args->input_stream, src_data, 0, datasize);
        if (status != HAILO_SUCCESS) {
            printf("-E- hailo_stream_sync_write_all_raw_buffer failed: %d",status);
            break;
        }
    }
    if (write_args->write_log==1)
        fclose(fp);

l_release_buffers:
    FREE(src_data);
l_exit:
    write_args->status = status;
    return NULL;
}

void* _recv_thread(void *args) {
    hailo_status status = HAILO_SUCCESS;
    recv_thread_args_t *recv_args = (recv_thread_args_t *)args;
    uint32_t datasize = recv_args->host_frame_size;
    struct timespec ts;
    FILE *fp;
    uint8_t *dst_data = (uint8_t*)malloc(datasize);
    unsigned lat_counter = 0;
    uint32_t flag_100 = 0;

    printf("-I- Recv thread %d started, datasize: %d\n", recv_args->tid, datasize);
    if (recv_args->write_log==1) {
        char *log_name = (char*)malloc(20 * sizeof(char));
        sprintf(log_name, "rx_tid_%d.log", recv_args->tid);
        fp = fopen(log_name, "w");
        free(log_name);
    }
    flag_100 = (uint32_t)recv_args->num_images / 100;
    if (flag_100==0)
        flag_100 = 1;
    
    actual_measurments = 0;

    for (uint32_t j = 1; j <= (uint32_t)recv_args->num_images; j++) {
        status = hailo_stream_sync_read_all_raw_buffer(recv_args->output_stream, dst_data, 0, datasize);
        REQUIRE_SUCCESS(status, l_exit, "hailo_stream_sync_read_all_raw_buffer failed");
        
        if ((j % flag_100==0) && (lat_counter < LATENCY_MEASUREMENTS)) {
            clock_gettime(CLOCK_REALTIME, &ts);
            printf("-I- [%10ld.%ld s] TID:%d Recv [%3d/%3d] 0x%x%x%x%x\n", (long)ts.tv_sec, ts.tv_nsec/1000000, recv_args->tid, j, recv_args->num_images, dst_data[0], dst_data[1], dst_data[2], dst_data[3]);
            recv_clock_t[recv_args->tid][lat_counter++] = ts;
            actual_measurments++;
        }
        if (recv_args->write_log==1)
            fprintf(fp, "%d Recv [%10ld.%ld s] %3d/%3d %x\n", recv_args->tid, (long)ts.tv_sec, ts.tv_nsec/1000000, j, recv_args->num_images, *dst_data);
    }
    if (recv_args->write_log==1)
        fclose(fp);
    
l_exit:
    free(dst_data);
    recv_args->status = status;
    return NULL;
}

hailo_status infer(
    hailo_input_stream input_stream, hailo_output_stream *output_streams, int output_stream_cnt, int num_images, int write_log, 
    size_t* host_output_frame_size, size_t host_input_frame_size, hailo_stream_info_t *all_streams_info, int source_fps)
{
    hailo_status status = HAILO_SUCCESS;
    pthread_t write_thread = 1;
    recv_thread_args_t recv_args[NOF_STREAMS-1];
    pthread_t recv_threads[NOF_STREAMS-1];

    write_thread_args_t write_args = {
        .input_stream = input_stream,
        .status = HAILO_UNINITIALIZED,
        .output_streams_cnt = output_stream_cnt,
        .num_images = num_images,
        .host_frame_size = host_input_frame_size,
        .write_log = write_log,
        .frame_hw_size = all_streams_info[0].hw_frame_size,
        .tid = 0,
        .source_fps = source_fps
    };
    for (int ii=0; ii<output_stream_cnt; ii++) {
        recv_args[ii].output_stream = output_streams[ii];
        recv_args[ii].tid = ii;
        recv_args[ii].status = HAILO_UNINITIALIZED;
        recv_args[ii].num_images = num_images;
        recv_args[ii].write_log = write_log;
        recv_args[ii].frame_hw_size = all_streams_info[ii+1].hw_frame_size;
        recv_args[ii].host_frame_size = host_output_frame_size[ii];
        (void) pthread_create(&recv_threads[ii], NULL, _recv_thread, &recv_args[ii]);
    };

    (void) pthread_create(&write_thread, NULL, _send_thread, &write_args);
    pthread_join(write_thread, NULL);

    if (HAILO_SUCCESS != write_args.status) {
        printf("-E- write_thread failed\n");
        status = HAILO_INTERNAL_FAILURE;
    }
    for (int ii=0; ii<output_stream_cnt; ii++) {
        pthread_join(recv_threads[ii], NULL);
        printf("-I- Closing Recv thread #%d: %d\n", recv_args[ii].tid, recv_args[ii].status);
        if (HAILO_SUCCESS != recv_args[ii].status) {
            printf("-E- Recv thread %d failed\n", recv_args[ii].tid);
            status = HAILO_INTERNAL_FAILURE;
        }
    }
    return status;
}

void print_net_banner(hailo_stream_info_t *all_stream_infos, int cnt) {
    printf(BOLDCYAN);
    printf("-I-----------------------------------------------\n");
    for (int ii=0; ii<=cnt; ii++) {
        printf("-I- %s[%d]: %s (%d, %d, %d)\n", get_direction_name(all_stream_infos[ii].direction), ii, all_stream_infos[ii].name, all_stream_infos[ii].shape.height, all_stream_infos[ii].shape.width,
            all_stream_infos[ii].shape.features);
    }
    printf("-I-----------------------------------------------\n");
    printf(RESET);
}

double get_time_from_ts(struct timespec ts) {
    double result = (double)(ts.tv_sec * 1000 + (double)(ts.tv_nsec / 1000000.0));
    return result;
}

double calc_latency(int count) {
    double result = 0;
    double cur_rcv;
    double cur_snd;

    for (int j=0; j<actual_measurments; j++) {
        cur_snd = get_time_from_ts(sent_clock_t[j]);
        cur_rcv = get_time_from_ts(recv_clock_t[0][j]);
        for (int ii=1; ii<count; ii++) {
            if (get_time_from_ts(recv_clock_t[ii][j]) > cur_rcv) {
                cur_rcv = get_time_from_ts(recv_clock_t[ii][j]);
            }
        }
        result += (cur_rcv - cur_snd);
    }
    return result/actual_measurments;
}

void print_inference_stats(struct timespec start_time, struct timespec end_time, hailo_stream_info_t *all_stream_infos, int output_stream_cnt, int num_images, uint32_t latency_readout)
{
    double start_time_secs = (double)start_time.tv_sec + ((double)start_time.tv_nsec / NSEC_IN_SEC);
    double end_time_secs = (double)end_time.tv_sec + ((double)end_time.tv_nsec / NSEC_IN_SEC);
    double infer_time_secs = end_time_secs - start_time_secs;
    static float mbit_per_byte = 8.0f / 1024.0f / 1024.0f;
    uint32_t send_frame_size = all_stream_infos[0].hw_frame_size;
    uint32_t recv_frame_size = 0;

    printf(BOLDGREEN);
    printf("-I-----------------------------------------------\n");
    printf("-I- Total time:        %4.2lf sec\n", infer_time_secs);
    printf("-I- Average FPS:       %4.2lf\n", num_images / infer_time_secs);
    if (num_images==1)
	    printf("-I- Average Latency:   %3.2lf ms\n", calc_latency(output_stream_cnt));

    printf("-I- Send data rate:    %-4.2lf Mbit/s\n",
        (double)(num_images) * send_frame_size * mbit_per_byte / infer_time_secs);
    for (int i=1; i<=output_stream_cnt; i++) {
        recv_frame_size = all_stream_infos[i].hw_frame_size;
        printf("-I- Recv[%d] data rate: %-4.2lf Mbit/s\n", i,
            (double)(num_images) * recv_frame_size * mbit_per_byte / infer_time_secs);
        printf("-I-----------------------------------------------\n");
    }
    printf(RESET);
}

const char** get_jlf_files_form_path(const char *dir_name, uint8_t *actual_number_of_jlfs_files) {
    static char jlf_files[HAILO_MAX_NUMBER_OF_JLFS][PATH_MAX];
    static const char *res_jlf_files[HAILO_MAX_NUMBER_OF_JLFS];
    DIR *dir = NULL;
    struct dirent *entry = NULL;
    uint8_t i = 0;

    dir = opendir(dir_name);
    if (NULL == dir) {
        return NULL;
    }

    entry = readdir(dir);
    while (NULL != entry) {
        if (entry->d_name[0] != '.') {
            (void)snprintf(jlf_files[i], sizeof(jlf_files[i]), "%s%s", dir_name, entry->d_name);
            res_jlf_files[i] = jlf_files[i];
            i++;
        }
        entry = readdir(dir);
    }
    (void) closedir(dir);
    *actual_number_of_jlfs_files = i;
    return res_jlf_files;
}

void set_default_eth_output_stream_params(hailo_eth_output_stream_params_t* stream) {
    stream->base_params.transform_mode = HAILO_STREAM_TRANSFORM_COPY;
    stream->base_params.user_buffer_format.order = HAILO_FORMAT_ORDER_NHWC;
    stream->base_params.user_buffer_format.flags = HAILO_FORMAT_FLAGS_QUANTIZE;
    stream->base_params.user_buffer_format.type = HAILO_FORMAT_TYPE_UINT8;
    stream->device_port = 0;
    stream->host_address.sin_family = AF_INET;
    stream->host_address.sin_port = 0;
    stream->host_address.sin_addr.s_addr = INADDR_ANY;
    // stream->is_sync_enabled = true;
    memset(&stream->host_address.sin_zero, 0, sizeof(stream->host_address.sin_zero));
}

void set_default_pcie_output_stream_params(hailo_pcie_output_stream_params_t* stream) {
    stream->base_params.transform_mode = HAILO_STREAM_TRANSFORM_COPY;
    stream->base_params.user_buffer_format.order = HAILO_FORMAT_ORDER_NHWC;
    stream->base_params.user_buffer_format.flags = HAILO_FORMAT_FLAGS_QUANTIZE;
    stream->base_params.user_buffer_format.type = HAILO_FORMAT_TYPE_UINT8;
}

hailo_status create_and_load_jlfs(hailo_device device, hailo_jlf *jlf, char* jlf_dir)
{
    hailo_status status = HAILO_SUCCESS;
    const char **jlf_files = NULL;
    uint8_t actual_number_of_jlf_files = 0;
    uint8_t jlf_buffer[96*1024];


    jlf_files = get_jlf_files_form_path(jlf_dir, &actual_number_of_jlf_files);
    if (NULL == jlf_files) {
        printf("Failed to get jlf files from path\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_exit;
    }

    status = hailo_create_jlf_files(jlf_files, actual_number_of_jlf_files, jlf_buffer, sizeof(jlf_buffer), jlf);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create jlf files");

    status = hailo_configure_device_from_jlf(device, *jlf, jlf_buffer, sizeof(jlf_buffer));
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to configure device from jlf");

    status = HAILO_SUCCESS;
    goto l_exit;
    
l_release_jlf:
    (void) hailo_release_jlf(*jlf);
l_exit:
    return status;
}

hailo_status initialize_eth_device(const char *interface_name, hailo_eth_device_info_t *eth_device_info,
    hailo_device *device)
{
    hailo_status status = HAILO_UNINITIALIZED;
    hailo_eth_device_info_t eth_device_info1 = {0};
    size_t number_of_devices = 0;
    
    status = hailo_scan_ethernet_devices(interface_name, &eth_device_info1, 1, &number_of_devices, HAILO_DEFAULT_ETH_SCAN_TIMEOUT_MS);
    REQUIRE_SUCCESS(status, l_exit, "Failed to scan for eth_devices");
    if (0 == number_of_devices) {
        printf("No eth_device found on the given interface\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_exit;
    }

    status = hailo_create_ethernet_device(&eth_device_info1, device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create eth_device");
    printf("DEBUG: finished initialize_eth_device\n");

    status = HAILO_SUCCESS;
l_exit:
    return status;
}

hailo_status initialize_pcie_device(hailo_pcie_device_info_t *pcie_device_info, hailo_device *device)
{
    hailo_status status = HAILO_UNINITIALIZED;
    size_t number_of_devices = 0;

    status = hailo_scan_pcie_devices(pcie_device_info, 2, &number_of_devices);
    REQUIRE_SUCCESS(status, l_exit, "Failed to scan for pcie_devices");

    if (0 == number_of_devices) {
        printf("No pcie_device found\n");
        status = HAILO_INTERNAL_FAILURE;
        goto l_exit;
    }

    status = hailo_create_pcie_device(pcie_device_info, device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create pcie_device");

    status = HAILO_SUCCESS;
l_exit:
    return status;
}

hailo_status create_and_activate_eth_streams(hailo_device device, hailo_jlf jlf, hailo_input_stream *input_stream,
    hailo_stream_info_t *streams_info, size_t* host_input_frame_size, hailo_output_stream *output_streams, 
    size_t* host_output_frame_size, int* output_stream_cnt)
{
    hailo_status status = HAILO_SUCCESS;
    hailo_eth_input_stream_params_t eth_input_stream_params = HAILO_ETH_INPUT_STREAM_PARAMS_DEFAULT;
    hailo_eth_output_stream_params_t eth_output_stream_params[NOF_STREAMS];
    hailo_stream_info_t all_stream_infos[NOF_STREAMS] = {};
    size_t number_of_streams = 0;

    status = hailo_jlf_get_all_stream_infos(jlf, all_stream_infos, sizeof(all_stream_infos), &number_of_streams);
    REQUIRE_SUCCESS(status, l_exit, "Failed to get eth_stream_infos");

    for (size_t i = 0; i < number_of_streams; i++) {
        streams_info[i] = all_stream_infos[i];
        if (HAILO_H2D_STREAM == all_stream_infos[i].direction) {
            // We need the following code the change the input format in case this is an FC layer
            if (all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NC ||
                all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NHW) {
                    eth_input_stream_params.base_params.user_buffer_format.order = all_stream_infos[i].format.order;
            }
            *host_input_frame_size = hailo_get_host_frame_size(&all_stream_infos[i], &eth_input_stream_params.base_params);
            status = hailo_create_eth_input_stream_from_jlf_by_index(device, jlf, streams_info[i].index,
                &eth_input_stream_params, input_stream);
            REQUIRE_SUCCESS(status, l_exit, "Failed to create eth_input_stream");
            status = hailo_activate_input_stream(device, *input_stream);
            REQUIRE_SUCCESS(status, l_release_streams, "Failed to activate input stream");
        } else if (HAILO_D2H_STREAM == all_stream_infos[i].direction) {
            set_default_eth_output_stream_params(&eth_output_stream_params[*output_stream_cnt]);
            if (all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NC ||
                all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NHW ||
                all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_HAILO_NMS) {
                eth_output_stream_params[*output_stream_cnt].base_params.user_buffer_format.order = all_stream_infos[i].format.order;
            }
            host_output_frame_size[*output_stream_cnt] = hailo_get_host_frame_size(&streams_info[i], &eth_output_stream_params[*output_stream_cnt].base_params);
            status = hailo_create_eth_output_stream_from_jlf_by_index(device, jlf, streams_info[i].index,
                &eth_output_stream_params[*output_stream_cnt], &output_streams[*output_stream_cnt]);
            REQUIRE_SUCCESS(status, l_exit, "Failed to create eth_output_stream");
            status = hailo_activate_output_stream(device, output_streams[*output_stream_cnt]);
            REQUIRE_SUCCESS(status, l_release_streams, "Failed to activate output stream");
            *output_stream_cnt+=1;
        } else {
            printf("Invalid stream direction\n");
            status = HAILO_INTERNAL_FAILURE;
        }
    }

l_release_streams:
    if (status != HAILO_SUCCESS) {
        for (int ii=0;ii<*output_stream_cnt;ii++) 
            (void) hailo_release_output_stream(device, output_streams[ii]);
        (void) hailo_release_input_stream(device, input_stream);
    }
l_exit:
    return status;
}

hailo_status create_and_activate_pcie_streams(hailo_device device, hailo_jlf jlf, hailo_input_stream *input_stream,
    hailo_stream_info_t *streams_info, size_t* host_input_frame_size, hailo_output_stream *output_streams, 
    size_t* host_output_frame_size, int* output_stream_cnt)
{
    hailo_status status = HAILO_SUCCESS;
    hailo_pcie_input_stream_params_t pcie_input_stream_params = HAILO_PCIE_STREAM_PARAMS_DEFAULT;
    hailo_pcie_output_stream_params_t pcie_output_stream_params[NOF_STREAMS];
    hailo_stream_info_t all_stream_infos[NOF_STREAMS] = {};
    size_t number_of_streams = 0;

    status = hailo_jlf_get_all_stream_infos(jlf, all_stream_infos, sizeof(all_stream_infos), &number_of_streams);
    REQUIRE_SUCCESS(status, l_exit, "Failed to get pcie_stream_infos");

    for (size_t i = 0; i < number_of_streams; i++) {
        streams_info[i] = all_stream_infos[i];
        if (HAILO_H2D_STREAM == all_stream_infos[i].direction) {
            // We need the following code the change the input format in case this is an FC layer
            if (all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NC ||
                all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NHW) {
                    pcie_input_stream_params.base_params.user_buffer_format.order = all_stream_infos[i].format.order;
                    //pcie_input_stream_params.base_params.transform_mode = HAILO_STREAM_NO_TRANSFORM;
            }
            *host_input_frame_size = hailo_get_host_frame_size(&all_stream_infos[i], &pcie_input_stream_params.base_params);
            status = hailo_create_pcie_input_stream_from_jlf_by_index(device, jlf, streams_info[i].index,
                &pcie_input_stream_params, input_stream);
            REQUIRE_SUCCESS(status, l_exit, "Failed to create pcie_input_stream");
            status = hailo_activate_input_stream(device, *input_stream);
            REQUIRE_SUCCESS(status, l_release_streams, "Failed to activate input stream");
        } else if (HAILO_D2H_STREAM == all_stream_infos[i].direction) {
            set_default_pcie_output_stream_params(&pcie_output_stream_params[*output_stream_cnt]);
            if (all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NC ||
                all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_NHW ||
                all_stream_infos[i].format.order == HAILO_FORMAT_ORDER_HAILO_NMS) {
                pcie_output_stream_params[*output_stream_cnt].base_params.user_buffer_format.order = all_stream_infos[i].format.order;
            }
            host_output_frame_size[*output_stream_cnt] = hailo_get_host_frame_size(&streams_info[i], &pcie_output_stream_params[*output_stream_cnt].base_params);
            status = hailo_create_pcie_output_stream_from_jlf_by_index(device, jlf, streams_info[i].index,
                &pcie_output_stream_params[*output_stream_cnt], &output_streams[*output_stream_cnt]);
            REQUIRE_SUCCESS(status, l_exit, "Failed to create pcie_output_stream");
            status = hailo_activate_output_stream(device, output_streams[*output_stream_cnt]);
            REQUIRE_SUCCESS(status, l_release_streams, "Failed to activate output stream");
            *output_stream_cnt+=1;
        } else {
            printf("Invalid stream direction\n");
            status = HAILO_INTERNAL_FAILURE;
        }
    }

l_release_streams:
    if (status != HAILO_SUCCESS) {
        for (int ii=0;ii<*output_stream_cnt;ii++) 
            (void) hailo_release_output_stream(device, output_streams[ii]);
        (void) hailo_release_input_stream(device, input_stream);
    }
l_exit:
    return status;
}

hailo_status initiazlie_pcie_inference(hailo_device *device, hailo_input_stream *input_stream, hailo_stream_info_t *streams_info,
    hailo_output_stream *output_streams, size_t *host_input_frame_size,
    size_t *host_output_frame_size, char* jlf_dir, int* output_stream_cnt)
{
    hailo_pcie_device_info_t pcie_device_info[2];// = {0};
    hailo_jlf jlf = NULL;
    hailo_status status = HAILO_UNINITIALIZED;

    status = initialize_pcie_device(pcie_device_info, device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to initialize pcie_device");
    
    status = create_and_load_jlfs(*device, &jlf, jlf_dir);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create and load jlfs");
    
    status = create_and_activate_pcie_streams(*device, jlf, input_stream, streams_info,
        host_input_frame_size, output_streams, host_output_frame_size, output_stream_cnt);
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to create and activate pcie_streams");
    
    status = HAILO_SUCCESS;
    goto l_exit;

l_release_jlf:
    (void) hailo_release_jlf(jlf);
l_exit:
    return status;
}

hailo_status initizlie_eth_inference(const char* interface_name, hailo_device *device, hailo_input_stream *input_stream, hailo_stream_info_t *streams_info,
    hailo_output_stream *output_streams, size_t *host_input_frame_size,
    size_t *host_output_frame_size, char* jlf_dir, int* output_stream_cnt)
{
    hailo_jlf jlf = NULL;
    hailo_status status = HAILO_UNINITIALIZED;
    hailo_eth_device_info_t *eth_device_info = {0};

    hailo_eth_input_stream_params_t eth_input_stream_params = HAILO_ETH_INPUT_STREAM_PARAMS_DEFAULT;
    hailo_eth_output_stream_params_t eth_output_stream_params = HAILO_ETH_OUTPUT_STREAM_PARAMS_DEFAULT;

    status = initialize_eth_device(interface_name, eth_device_info, device);
    REQUIRE_SUCCESS(status, l_exit, "Failed to initialize eth_device");
    
    status = create_and_load_jlfs(*device, &jlf, jlf_dir);
    REQUIRE_SUCCESS(status, l_exit, "Failed to create and load jlfs");
    
    status = create_and_activate_eth_streams(*device, jlf, input_stream, streams_info,
        host_input_frame_size, output_streams, host_output_frame_size, output_stream_cnt);
    REQUIRE_SUCCESS(status, l_release_jlf, "Failed to create and activate eth_streams");
    //*host_input_frame_size = hailo_get_host_frame_size(input_stream_info, &eth_input_stream_params.base_params);
    //*host_output_frame_size = hailo_get_host_frame_size(output_stream_info, &eth_output_stream_params.base_params);

    status = HAILO_SUCCESS;
    goto l_exit;

l_release_jlf:
    (void) hailo_release_jlf(jlf);
l_exit:
    return status;
}

int main(int argc, char **argv)
{
    hailo_status status = HAILO_UNINITIALIZED;
    hailo_device device = NULL;
    hailo_input_stream input_stream = NULL;
    hailo_output_stream output_streams[NOF_STREAMS];
    // hailo_eth_input_stream_params_t input_stream_params = HAILO_ETH_INPUT_STREAM_PARAMS_DEFAULT;
    // hailo_eth_output_stream_params_t output_streams_params[NOF_STREAMS-1];
    hailo_stream_info_t all_stream_infos[NOF_STREAMS];
    // hailo_stream_info_t output_streams_info[NOF_STREAMS-1];
    size_t host_input_frame_size = 0;
    size_t host_output_frame_size[NOF_STREAMS];
    struct timespec start_time = {0};
    struct timespec end_time = {0};
    size_t number_of_streams = 0;
    uint8_t input_fifo_index = 0;
    int output_stream_cnt = 0;
    uint8_t output_fifo_indexes[NOF_STREAMS-1];
    char *jlf_dir = "./JLFs/";
    int opt;
    char *iface = "pcie";
    int num_img = 100;
    int debug = 0;
    int write_log = 0;
    int source_fps = 0;
    uint32_t latency_readout = 0;

    while ((opt = getopt(argc, argv, "f:i:j:n:dl")) != -1) {
        switch (opt) {
            case 'j': jlf_dir = optarg; break;
            case 'i': iface =   optarg; break;
            case 'n': num_img = atoi(optarg); break;
            case 'f': source_fps = atoi(optarg); break;
            case 'l': write_log = 1; break;
            case '?': fprintf (stderr, "Option -%c requires an argument.\n", optopt);
            default:
              fprintf(stderr, "Usage: %s -i INTERFACE -j JLF-DIR [-n NUM-IMAGES]\n\n", argv[0]);
              fprintf(stderr, "     -i INTERFACE      The Interface, defaults to \'eno2\', can be also \'pcie\'\n");
              fprintf(stderr, "     -j JLF-DIR        The JLFs directory, defaults to \'./JLFs/\'\n");
              fprintf(stderr, "     -n NUM-IMAGES     The number of images to process, defaults to 100\n");
              fprintf(stderr, "     -f FPS            Emulate source FPS, defaults to infinity\n");
              fprintf(stderr, "     -l                Each receive thread will write a log file\n");
              exit(EXIT_FAILURE);
        }
    }
    printf(BOLDCYAN);
    printf("-I- Running on interface: %s %d images\n", iface, num_img);
    printf("-I- Reading JLFs from: %s\n", jlf_dir);
    printf(RESET);
    
    if (strncmp("pcie", iface, 4) == 0) {
        status = initiazlie_pcie_inference(&device, &input_stream, all_stream_infos, output_streams, &host_input_frame_size, 
            host_output_frame_size, jlf_dir, &output_stream_cnt);
    } else {
        status = initizlie_eth_inference(iface, &device, &input_stream, all_stream_infos, output_streams, &host_input_frame_size, 
            host_output_frame_size, jlf_dir, &output_stream_cnt);        
    }
    REQUIRE_SUCCESS(status, l_exit, "Failed to scan for PCIe devices");

    print_net_banner(all_stream_infos, output_stream_cnt);

    (void) clock_gettime(CLOCK_MONOTONIC, &start_time);    
    status = infer(input_stream, output_streams, output_stream_cnt, num_img, write_log, host_output_frame_size, host_input_frame_size, all_stream_infos, source_fps);
    
    #ifdef FW_LAT
    status = hailo_latency_measurement_read(device, &latency_readout);
    #endif
    REQUIRE_SUCCESS(status, l_release_output_stream, "Inference failure");
    (void) clock_gettime(CLOCK_MONOTONIC, &end_time);

    print_inference_stats(start_time, end_time, all_stream_infos, output_stream_cnt, num_img, latency_readout);
    status = HAILO_SUCCESS;

l_release_output_stream:
    for (int i=0;i<output_stream_cnt;i++)
        (void) hailo_release_output_stream(device, output_streams[i]);

l_release_input_stream:
    (void) hailo_release_input_stream(device, input_stream);
l_release_device:
    (void) hailo_release_device(device);
l_exit:
    printf(MAGENTA);
    printf("-I- Finished gracefully\n");
    printf(RESET);
    return status;
}
