package demo

import (
	"context"
	"errors"
	"fmt"
	"go.opentelemetry.io/contrib/instrumentation/net/http/otelhttp"
	"go.opentelemetry.io/otel"
	"io/ioutil"
	"net/http"
	"time"
)

var (
	// ErrMalformedEntity indicates malformed entity specification (e.g. invalid username or password).
	ErrMalformedEntity    = errors.New("malformed entity specification")
	ErrConflict           = errors.New("conflict")
	ErrUnauthorizedAccess = errors.New("ErrUnauthorizedAccess")
	ErrForbidden          = errors.New("ErrForbidden")
)

var loc, _ = time.LoadLocation("Asia/Ho_Chi_Minh")

type Service interface {
	Call(ctx context.Context) (interface{}, error)
}

var _ Service = (*demoOtelService)(nil)

type demoOtelService struct {
}

func (svc demoOtelService) Call(ctx context.Context) (interface{}, error) {

	ctx, span := otel.Tracer("service-demo-call").Start(ctx, "Demo-Call")
	fmt.Printf("Call\n")
	// URL of the server endpoint
	url := "http://localhost:8002/demo2-otel/call"

	// Create a new HTTP client
	client := http.Client{Transport: otelhttp.NewTransport(http.DefaultTransport)}

	// Create a new HTTP request (POST)
	req, err := http.NewRequestWithContext(ctx, http.MethodPost, url, nil) // nil for no body
	if err != nil {
		panic(err)
	}

	// Set headers if needed (optional)
	// req.Header.Set("Content-Type", "application/json") // Example header

	// Send the request and get the response
	resp, err := client.Do(req)
	if err != nil {
		panic(err)
	}
	defer resp.Body.Close()

	// Read the response body (optional)
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		panic(err)
	}

	// Print the response status code and body (if read)
	fmt.Println("Status Code:", resp.StatusCode)
	if body != nil {
		fmt.Println("Response Body:", string(body))
	}
	defer span.End()
	return "OK", nil
}

func NewNoDb() Service {
	return &demoOtelService{}
}

type Response struct {
	Code int         `json:"code"`
	Msg  string      `json:"msg"`
	Data interface{} `json:"data,omitempty"`
}
