package com.securityandsafetythings.examples.aiapp.manager;

import android.app.Application;
import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import com.securityandsafetythings.examples.aiapp.RestEndPoint;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiprocess.AILicencePlateProcessor;
import com.securityandsafetythings.examples.aiapp.aicore.aiprocess.AIOCRProcessor;
import com.securityandsafetythings.examples.aiapp.aicore.aiprocess.AIProcess;
import com.securityandsafetythings.examples.aiapp.aicore.aiprocess.AIVehicleProcessor;
import com.securityandsafetythings.examples.aiapp.aicore.aiprocess.aicallback.AiProcessCallback;
import com.securityandsafetythings.examples.aiapp.aicore.mediapool.MediaNode;

import org.jetbrains.annotations.NotNull;

import java.util.HashMap;
import java.util.Map;

public class AIManager {
    static final String LOGTAG = AIManager.class.getSimpleName();
    private final Context context;
    private final Application application;
    private final RestEndPoint mRestEndPoint;

    private AlgoType currentAlgoType;
    private AIProcess firstAIProcessor;

    private final Map<AlgoType, AIProcess> listAlgoProcessors;

    public enum AlgoType {
        VEHICLE_DETECT      ("VehicleDetection"),
        LPD                 ("LicencePlateDetection"),
        NONE                ("None"),
        LPD_LPR             ("LPD&LPR"),
        LPR                 ("LPR"),
        VHD_LPD_LPR         ("VHD&LPD&LPR");

        String algoType;
        AlgoType(String algoType) {
            this.algoType = algoType;
        }
    }

    public AIManager(Context context, Application application, RestEndPoint mRestEndPoint) {
        this.listAlgoProcessors = new HashMap<>();
        this.context = context;
        this.application = application;
        this.mRestEndPoint = mRestEndPoint;
    }

    /** This method called to setup Algorithm feature by
     * Safely stop all Process and free ai-models then change to new algorithm process
     * */
    public void setAlgorithm(@NonNull AlgoType algoType){
        /* End all process in current process map */
        if ( !listAlgoProcessors.isEmpty() ) {
            for (Map.Entry<AlgoType, AIProcess> processEntry : listAlgoProcessors.entrySet()) {
                try {
                    processEntry.getValue().endProcess().join();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            Log.d(LOGTAG + "_checkChangeAlgo", "finish release ai-process");
        }
        /* Clear all process after completely terminated */
        listAlgoProcessors.clear();
        firstAIProcessor = null;

        /* load new ai-process */
        loadAlgorithms(algoType);

        /* Start all new added process on map */

        for (Map.Entry<AlgoType, AIProcess> processEntry : listAlgoProcessors.entrySet()) {
            Log.d(LOGTAG,"Start process: " + processEntry.getValue());
            processEntry.getValue().start();
        }
        currentAlgoType = algoType;
    }

    private void loadAlgorithms(@NotNull AlgoType algoType){
        Log.d(LOGTAG, "loadAlgorithms for mode = " + algoType);
        /* setup AiProcess pipeline - define pipeline and setup first aiProcess */
        switch (algoType) {
            case NONE: {
                // change current algoType
                currentAlgoType = AlgoType.NONE;

            }
            case LPR:{
                AiProcessCallback aiProcessCallback = new AiProcessCallback() {
                    @Override
                    public void onAiProcessDetectNothing(InferenceResult inferenceResult) {
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {
                        mRestEndPoint.setImage(outputInference.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessTerminate() {
                    }
                };
                AIOCRProcessor aiocrProcessor = new AIOCRProcessor(context, application, null);
                aiocrProcessor.setAIProcessCallback(aiProcessCallback);
                firstAIProcessor = aiocrProcessor;
                break;
            }

            case LPD: {
                currentAlgoType = AlgoType.LPD;
                AiProcessCallback aiProcessCallback = new AiProcessCallback() {
                    @Override
                    public void onAiProcessDetectNothing(InferenceResult inferenceResult) {
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {
                        mRestEndPoint.setImage(outputInference.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessTerminate() {
                    }
                };
                AILicencePlateProcessor aiLicencePlateProcessor = new AILicencePlateProcessor(context, application, null);
                aiLicencePlateProcessor.setAIProcessCallback(aiProcessCallback);
                firstAIProcessor = aiLicencePlateProcessor;
                break;
            }
            case LPD_LPR:{
                currentAlgoType = AlgoType.LPD_LPR;
                AiProcessCallback aiProcessCallback1 = new AiProcessCallback() {
                    @Override
                    public void onAiProcessDetectNothing(InferenceResult inferenceResult) {
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {
                        mRestEndPoint.setImage(outputInference.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessTerminate() {
                    }
                };
                AiProcessCallback aiProcessCallback2 = new AiProcessCallback() {
                    @Override
                    public void onAiProcessDetectNothing(InferenceResult inferenceResult) {
                        mRestEndPoint.setRTSPFrame(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {
                        mRestEndPoint.setRTSPFrame(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessTerminate() {
                    }
                };

                AIOCRProcessor aiocrProcessor = new AIOCRProcessor(context, application, null);
                aiocrProcessor.setAIProcessCallback(aiProcessCallback2);
                AILicencePlateProcessor aiLicencePlateProcessor = new AILicencePlateProcessor(context, application, aiocrProcessor);
                aiLicencePlateProcessor.setAIProcessCallback(aiProcessCallback1);

                // define pipeline here ...
                listAlgoProcessors.put(AlgoType.LPD, aiLicencePlateProcessor);
                listAlgoProcessors.put(AlgoType.LPR, aiocrProcessor);

                // Setup first aiProcess here ...
                firstAIProcessor = aiLicencePlateProcessor;
                break;
            }
            case VEHICLE_DETECT: {
                // change current algoType
                currentAlgoType = AlgoType.VEHICLE_DETECT;

                // define pipeline here ...
                AiProcessCallback aiProcessCallback = new AiProcessCallback() {
                    @Override
                    public void onAiProcessDetectNothing(InferenceResult inferenceResult) {
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {
                        //ImageUtils.saveBitmapToCache(context, inferenceResult.getProcessBitmap(), "outputBitmap");
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessTerminate() {

                    }
                };
                AIVehicleProcessor aiVehicleProcessor = new AIVehicleProcessor(context, application, null);
                aiVehicleProcessor.setAIProcessCallback(aiProcessCallback);
                listAlgoProcessors.put(AlgoType.VEHICLE_DETECT, aiVehicleProcessor);

                // Setup first aiProcess here ...
                firstAIProcessor = aiVehicleProcessor;
                break;
            }
            case VHD_LPD_LPR:{
                currentAlgoType = AlgoType.VHD_LPD_LPR;
                // define pipeline here ...
                AiProcessCallback aiProcessCallback = new AiProcessCallback() {
                    @Override
                    public void onAiProcessDetectNothing(InferenceResult inferenceResult) {
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {
                        //ImageUtils.saveBitmapToCache(context, inferenceResult.getProcessBitmap(), "outputBitmap");
                        mRestEndPoint.setImage(inferenceResult.getProcessBitmap());
                    }

                    @Override
                    public void onAiProcessTerminate() {

                    }
                };

                AIOCRProcessor aiocrProcessor                   = new AIOCRProcessor(context, application, null);
                AILicencePlateProcessor aiLicencePlateProcessor = new AILicencePlateProcessor(context, application, aiocrProcessor);
                AIVehicleProcessor aiVehicleProcessor           = new AIVehicleProcessor(context, application, aiLicencePlateProcessor);
                aiocrProcessor.setAIProcessCallback(aiProcessCallback);

                listAlgoProcessors.put(AlgoType.VEHICLE_DETECT, aiVehicleProcessor);
                listAlgoProcessors.put(AlgoType.LPD, aiLicencePlateProcessor);
                listAlgoProcessors.put(AlgoType.LPR, aiocrProcessor);

                // Setup first aiProcess here ...
                firstAIProcessor = aiVehicleProcessor;
                break;
            }
        }
    }

    /** This method called to push frame an existed pipeline which has been setup on loadAlgorithms()
     * Make sure pipeline has been set and firstAIProcessor not null
     * @param inputMedia*/
    public void addItemOnPipeline(MediaNode inputMedia){
        //aiProcessManagerCallBack.onGettingNewMediaNode(inputMedia);

        if (firstAIProcessor != null) {
            try {
                firstAIProcessor.addItem(inputMedia);
            } catch (InterruptedException e) {
                e.printStackTrace();
                //Log.d("MediaPool_", "free bmp =" + inputMedia.getBitmap().hashCode() + " on exception in addItemOnPipeline");
                inputMedia.free();
            }
        } else {
            //Log.d("MediaPool_", "free bmp =" + inputMedia.getBitmap().hashCode() + " on firstAIProcessor null");
            inputMedia.free();
        }
    }

    public void addItemOnPipeline(InferenceResult inferenceResult){
        //aiProcessManagerCallBack.onGettingNewMediaNode(inputMedia);

        if (firstAIProcessor != null) {
            try {
                firstAIProcessor.addItem(inferenceResult);
            } catch (InterruptedException e) {
                e.printStackTrace();
                //Log.d("MediaPool_", "free bmp =" + inputMedia.getBitmap().hashCode() + " on exception in addItemOnPipeline");
                inferenceResult.getProcessMedia().free();
            }
        } else {
            //Log.d("MediaPool_", "free bmp =" + inputMedia.getBitmap().hashCode() + " on firstAIProcessor null");
            inferenceResult.getProcessMedia().free();
        }
    }

}
