/*
 * Copyright 2019-2020 by Security and Safety Things GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.securityandsafetythings.examples.aiapp.utilities;

import static com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult.ResultName.licencePlateDetect;
import static com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult.ResultName.ocr;

import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.RectF;

import androidx.annotation.NonNull;

import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.Bbox;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.licensePlate.LPDResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.vehicle.VDResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.extractor.OCRResult;

import java.util.List;
import java.util.Map;

/**
 * Utility class to aid in rendering data
 */
public final class Renderer {
    private static final Paint BACKGROUND_PAINT = getBackgroundPaint();
    private static final int COLOR_RED = 0xffff0000;
    private static final int COLOR_YELLOW = 0xffffcc00;
    private static final int COLOR_GREEN = 0xff00ff00;
    private static final int COLOR_BLUE = 0xff0000ff;
    private static final int COLOR_WHITE = 0xffffffff;
    private static final int COLOR_BLACK = 0xff000000;

    private Renderer() {
    }

    @NonNull
    private static Paint getBackgroundPaint() {
        final Paint res = new Paint();
        res.setStyle(Paint.Style.STROKE);
        res.setColor(0);
        return res;
    }

    private static Paint getTextPaint() {
        final Paint res = new Paint();
        res.setStyle(Paint.Style.FILL);
        res.setTextSize(40);
        res.setColor(COLOR_YELLOW);//COLOR_WHITE//COLOR_RED
        return res;
    }

    private static Paint getTextPaint(int textSize, int color) {
        final Paint res = new Paint();
        res.setStyle(Paint.Style.FILL);
        res.setTextSize(textSize);
        res.setColor(color);//COLOR_WHITE//COLOR_RED
        return res;
    }

    private static Paint getLabelPaint(int textSize, int color) {
        final Paint res = new Paint();
        res.setStrokeWidth(0);
        res.setStyle(Paint.Style.FILL);

        res.setTextSize(textSize);
        res.setColor(color);//COLOR_WHITE//COLOR_RED
        return res;
    }

    private static Paint getBboxPaint(int size, int color) {
        final Paint res = new Paint();
        res.setStyle(Paint.Style.STROKE);
        res.setAlpha(10);
        res.setStrokeWidth(size);
        res.setColor(color);//COLOR_WHITE//COLOR_RED
        return res;
    }

    private static Paint getPointPaint(int size, int color){
        final Paint res = new Paint();
        res.setStyle(Paint.Style.FILL);
        res.setStrokeWidth(size);
        res.setColor(color);//COLOR_WHITE//COLOR_RED
        return res;
    }

    private static Paint getTextPaint(int color) {
        final Paint res = new Paint();
        res.setStyle(Paint.Style.FILL);
        res.setTextSize(2);
        res.setColor(color);//COLOR_WHITE//COLOR_RED
        return res;
    }

    public static void renderAllBoxVehicle(@NonNull Canvas canvas, VDResult vdResult, float fps){
        canvas.drawPaint(BACKGROUND_PAINT);

        Map<String, List<Bbox>> mapResults = vdResult.getBboxMapResult();

        int boxColor1 = COLOR_RED;

        for (Map.Entry<String, List<Bbox>> entry : mapResults.entrySet()){
            for (Bbox box : entry.getValue()){
                //Log.d("checkRender", "box = " + box.label);

                RectF rawBox = box.getLocation();
                float roundConf = (float)((int)(box.getConfidence() * 1000f)) / 1000f;
                canvas.drawText(box.label + "_" + roundConf,
                        rawBox.left + 1,
                        rawBox.top + 40f,
                        getTextPaint());
                canvas.drawRect(rawBox, getBboxPaint(40, COLOR_GREEN));

            }
        }

        fps = (float)((int)(fps * 1000f)) / 1000f;

        canvas.drawText(fps + "",
                10 + 1,
                10 + 40,
                getTextPaint(COLOR_YELLOW));
    }


    public static void renderPlate(@NonNull Canvas canvas, @NonNull LPDResult lpdResult, float fps) {
        canvas.drawPaint(BACKGROUND_PAINT);
        List<Bbox> bboxes = lpdResult.getBboxList();
        for (Bbox box : bboxes){
            RectF rawBox = box.getLocation();
            float roundConf = (float)((int)(box.getConfidence() * 1000f)) / 1000f;
            canvas.drawText(box.label + "_" + roundConf,
                    rawBox.left + 1,
                    rawBox.top - 40f,
                    getTextPaint());
            canvas.drawRect(rawBox, getBboxPaint(80, COLOR_GREEN));

            for (int j = 0; j < 5; ++j) {
                canvas.drawCircle(
                        (float)box.landmarks[j].x,
                        (float)box.landmarks[j].y,
                        2.0f,
                        getBboxPaint(80, COLOR_GREEN)
                );
            }

        }

        fps = (float)((int)(fps * 1000f)) / 1000f;

        canvas.drawText(fps + "",
                10 + 1,
                10 + 40,
                getTextPaint(COLOR_YELLOW));

    }

    public static void renderAllBoxVehicleOCRPlate(@NonNull Canvas canvas, @NonNull VDResult vdResult, float fps) {
        //canvas.drawPaint(BACKGROUND_PAINT);
        Map<String, List<Bbox>> vdResultMap = vdResult.getBboxMapResult();


        for (Map.Entry<String, List<Bbox>> vehicleBboxListEntry : vdResultMap.entrySet()) {
            List<Bbox> vehicleBboxList = vehicleBboxListEntry.getValue();

            for (Bbox boxVehicle : vehicleBboxList){
                //Log.d("checkRender", "box = " + box.label);

                //Render vehicle bbox, class, score
                RectF rawBoxVehicle = boxVehicle.getLocation();
                float roundConfVehicle = (float)((int)(boxVehicle.getConfidence() * 1000f)) / 1000f;
                canvas.drawText(boxVehicle.label + "_" + roundConfVehicle,
                        rawBoxVehicle.left + 1,
                        rawBoxVehicle.top - 40,
                        getTextPaint(80, COLOR_YELLOW));
                canvas.drawRoundRect(rawBoxVehicle,45, 45, getBboxPaint(15, COLOR_GREEN));

                LPDResult lpdResult = (LPDResult) boxVehicle.getInferenceResult(licencePlateDetect);

                if (!lpdResult.getBboxList().isEmpty()){
                    Bbox plateBbox = lpdResult.getBboxList().get(0);

                    for (int j = 0; j < 5; ++j) {
                        canvas.drawCircle(
                                (float)plateBbox.landmarks[j].x,
                                (float)plateBbox.landmarks[j].y,
                                3.0f,
                                getPointPaint(10, COLOR_RED)
                        );
                    }

                    OCRResult ocrResult = (OCRResult) boxVehicle.getInferenceResult(ocr);
                    String plate = ocrResult != null ? ocrResult.getLicencePlate() : "NONE";

                    RectF rawBoxPlate = plateBbox.getLocation();
                    float roundConfPlate = (float) ((int) (plateBbox.getConfidence() * 1000f)) / 1000f;
                    canvas.drawText( plate + "_" + roundConfPlate,
                            rawBoxPlate.left + 1,
                            rawBoxPlate.top - 80f,
                            getTextPaint(160, COLOR_GREEN));
                    canvas.drawRoundRect(rawBoxPlate, 5, 5, getBboxPaint(5, COLOR_BLUE));
                }
            }
        }
        fps = (float)((int)(fps * 1000f)) / 1000f;

        canvas.drawText(fps + "",
                10 + 1,
                10 + 40,
                getTextPaint(COLOR_YELLOW));


    }
}
