/*
 * Copyright 2019-2020 by Security and Safety Things GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.securityandsafetythings.examples.aiapp.services;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.media.Image;
import android.media.ImageReader;
import android.util.Log;

import androidx.annotation.NonNull;

import com.qualcomm.qti.snpe.NeuralNetwork;
import com.securityandsafetythings.app.VideoService;

import com.securityandsafetythings.examples.aiapp.R;
import com.securityandsafetythings.examples.aiapp.RestEndPoint;

import com.securityandsafetythings.examples.aiapp.UNITTEST.TestModelVehicleDetect;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.Bbox;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.face.FDResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.face.RetinaFaceDetector;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.extractor.MobileFaceExtractorV2;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.algorithm.nativewarpper.align.FaceAligner;
import com.securityandsafetythings.examples.aiapp.aicore.mediapool.MediaNode;
import com.securityandsafetythings.examples.aiapp.manager.AIManager;
import com.securityandsafetythings.examples.aiapp.utilities.BitmapUtilities;
import com.securityandsafetythings.video.RefreshRate;
import com.securityandsafetythings.video.VideoCapture;
import com.securityandsafetythings.video.VideoManager;
import com.securityandsafetythings.video.VideoSession;
import com.securityandsafetythings.webserver.RestHandler;
import com.securityandsafetythings.webserver.WebServerConnector;


import org.bytedeco.javacpp.Loader;
import org.bytedeco.javacpp.opencv_java;
import org.opencv.android.Utils;
import org.opencv.core.Mat;
import org.opencv.imgproc.Imgproc;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 *
 * This class responds to {@link #onCreate()} and {@link #onDestroy()} methods of the application
 * lifecycle. In order to receive images from the Video Pipeline, {@link MainService} extends
 * {@link VideoService} and implements its callbacks.
 *
 * The three callbacks are:
 *
 * 1. {@link #onVideoAvailable(VideoManager)} - triggered when the Video Pipeline is ready to begin transferring {@link Image} objects.
 * 2. {@link #onImageAvailable(ImageReader)} - triggered when the {@link ImageReader} acquires a new {@link Image}.
 * 3. {@link #onVideoClosed(VideoSession.CloseReason)} - triggered for one of the four close reasons mentioned in method's JavaDoc.
 *
 */
public class MainService extends VideoService {
    static final String LOGTAG = MainService.class.getSimpleName();

    private WebServerConnector mWebServerConnector;
    private RestEndPoint mRestEndPoint;

    private long previousTime = System.currentTimeMillis();

    private AIManager aiManager;
    private Bitmap inputBitmapCache;

    private RetinaFaceDetector retinaFaceDetector;
    private FaceAligner faceAligner;
    private MobileFaceExtractorV2 mobileFaceExtractorV2;

    //image queue for gw
    private final static int COMMAND_HANDLER_WATCHDOG_TIMER_MS = 2 * 60 * 1000;

    /**
     * {@link #onCreate()} initializes our {@link WebServerConnector}, {@link RestEndPoint}, and
     * {@link RestHandler}.
     *
     * The {@link WebServerConnector} acts as the bridge between our application and the webserver
     * which is contained in the Security and Safety Things SDK.
     *
     * The {@link RestEndPoint} is a class annotated with JaxRs endpoint annotations. This is the class
     * that we interact with via HTTP on the front end.
     *
     * The {@link RestHandler} acts as a wrapper class for our {@link RestEndPoint}. The Handler registers our
     * {@link RestEndPoint}, and connects it to the WebServer.
     *
     */
    @Override
    public void onCreate() {
        super.onCreate();

        Loader.load(opencv_java.class);
        System.loadLibrary("c++_shared");
        System.loadLibrary("Fr-align-lib");

        /*
         * Creates a RestHandler with a base path of 'app/getPackageName()'.
         */
        System.setProperty("org.bytedeco.javacpp.maxPhysicalBytes", "2048M");
        final String websiteAssetPath = "website";
        final RestHandler restHandler = new RestHandler(this, websiteAssetPath);
        /*
         * Causes static context serving to serve index.html for any path.
         */
        restHandler.serveIndexOnAnyRoute();
        /*
         * Registers the RestEndPoint with the server via the RestHandler class. The RestHandler
         * is just a wrapper for the RestEndPoint's JaxRs annotated functions.
         */
        mRestEndPoint = new RestEndPoint();
        restHandler.register(mRestEndPoint);
        /*
         * Connects the RestHandler with the WebServerConnector.
         */
        mWebServerConnector = new WebServerConnector(this);
        mWebServerConnector.connect(restHandler);

        aiManager = new AIManager(this, this.getApplication(), mRestEndPoint);
        aiManager.setAlgorithm(AIManager.AlgoType.VHD_LPD_LPR);

        retinaFaceDetector = new RetinaFaceDetector(getApplicationContext(), getApplication(), R.raw.retina_mb_nosm_h288_w512_quantized, NeuralNetwork.Runtime.DSP);
        faceAligner = new FaceAligner();
        mobileFaceExtractorV2 = new MobileFaceExtractorV2(getApplicationContext(), getApplication(), R.raw.frv4_mask, NeuralNetwork.Runtime.DSP);

//        Bitmap bitmap = BitmapFactory.decodeResource(getResources(), R.drawable.test);
//        inputBitmapCache = bitmap.copy(Bitmap.Config.ARGB_8888, true);

        //Bitmap bitmap = BitmapFactory.decodeResource(getResources(), R.drawable.img5m_sampletest_1);
        //inputBitmapCache = bitmap.copy( Bitmap.Config.ARGB_8888, true );

        //TODO: do some testcase here
        //TestModelVehicleDetect testModelVehicleDetect = new TestModelVehicleDetect(this, this.getApplication());
        //testModelVehicleDetect.testcase1();
    }

    /**
     * This callback is triggered when the Video pipeline is available to begin capturing images.
     *
     * @param manager is the {@link VideoManager} object we use to obtain access to the Video Pipeline
     */
    @SuppressWarnings("MagicNumber")
    @Override
    protected void onVideoAvailable(@NonNull final VideoManager manager) {
        /*
         * Gets a default VideoCapture instance which does not scale, rotate, or modify the image received from the Video Pipeline.
         */
        final VideoCapture capture = manager.getDefaultVideoCapture();
        /*
         * RefreshRate specifies how often the image should be updated, in this case 15 frames per second (FPS_15).
         */
        openVideo(capture, capture.getWidth(), capture.getHeight(), RefreshRate.FPS_15, false);
    }

    /**
     * Callback is triggered when an image is obtained, here images can be stored for retrieval from the exposed
     * methods of the {@link RestEndPoint}.
     *
     * @param reader is the {@link ImageReader} object we use to obtain still frames from the Video Pipeline
     */
    @Override
    protected void onImageAvailable(final ImageReader reader) {
        /*
         * Gets the latest image from the Video Pipeline and stores it in the RestEndPoint class so the
         * frontend can retrieve it via a GET call to rest/example/live.
         */
        try (Image image = reader.acquireLatestImage()) {
            // ImageReader may sometimes return a null image.
            if (image == null) {
                Log.e("onImageAvailable()", "ImageReader returned null image.");
                return;
            }
            long currentTime = System.currentTimeMillis();
            long deltaTime = currentTime - previousTime;
            float fps = 1000f / (float) (deltaTime > 0 ? deltaTime : 1) ;
            previousTime = currentTime;
            Log.d("checkPerform", "input fps = " + fps + " | frame size = " + image.getHeight() + "x" + image.getWidth());
            Bitmap imageBmp = BitmapUtilities.imageToBitmap(image);
            Bitmap copyBitmap = imageBmp.copy( Bitmap.Config.ARGB_8888, true );

            MediaNode inputMedia = new MediaNode(copyBitmap);

            Mat frameRGBA = new Mat();
            Mat frameRgb = new Mat();
            Utils.bitmapToMat(copyBitmap, frameRGBA);
            Imgproc.cvtColor(frameRGBA, frameRgb, Imgproc.COLOR_RGBA2RGB);

            Log.d(LOGTAG, "check input rgb mat = " + frameRgb.cols() + "x" + frameRgb.rows());

            InferenceResult inputInference = new InferenceResult(inputMedia, frameRgb) {
                @Override
                public ResultName getResultName() {
                    return null;
                }
            };

            FDResult fdResult = retinaFaceDetector.runInference(inputInference);

            for (Bbox bbox: fdResult.getBboxList()){
                Log.d(LOGTAG, "box: x1: " + bbox.x1+ "y1: " + bbox.y1 + "x2: "+ bbox.x2 + "y2: "+ bbox.y2);
            }

            long startFA = System.currentTimeMillis();

            FDResult faceAlign = (FDResult) faceAligner.runInference((InferenceResult) fdResult);

            long endFA = System.currentTimeMillis();
            float fpsFA = 1000f / (float) ((endFA-startFA) > 0 ? (endFA-startFA)  : 1) ;
            Log.d(LOGTAG + "_face aligner", "time face aligner: "+ fpsFA);

            for (Bbox box: faceAlign.getBboxList()){
                mobileFaceExtractorV2.runInference(box.getInferenceResult(InferenceResult.ResultName.faceAlign));
            }

//            aiManager.addItemOnPipeline(inputInference);

            mRestEndPoint.setImage(copyBitmap);
        }
    }

    /**
     * This callback would handle all tear-down logic, and is called when the {@link VideoSession} is stopped.
     * Four possible ways for the video session to be stopped are:
     *
     *  1. {@link VideoSession.CloseReason#SESSION_CLOSED}
     *  2. {@link VideoSession.CloseReason#VIRTUAL_CAMERA_CONFIGURATION_CHANGED}
     *  3. {@link VideoSession.CloseReason#VIRTUAL_CAMERA_CONFIGURATION_REMOVED}
     *  4. {@link VideoSession.CloseReason#RENDERING_FAILED}
     *
     * @param reason is the reason for closing the video pipeline
     */
    @Override
    protected void onVideoClosed(final VideoSession.CloseReason reason) {
        Log.i("Video Closed", reason.name());
    }

    /**
     * Disconnects the {@link WebServerConnector} when service is destroyed. This method is triggered when the application
     * is stopped.
     */
    @Override
    public void onDestroy() {
        mWebServerConnector.disconnect();
        super.onDestroy();
    }

    /** save bitmap for checking results */
    public void save_imageBit(String path, @NonNull Bitmap source){
        String fulpath = this.getFilesDir() + "/" + path + ".jpg";
        try {
            FileOutputStream fout = new FileOutputStream(fulpath);
            source.compress(Bitmap.CompressFormat.JPEG, 100, fout);
            fout.flush();
            fout.close();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

    }

}

