import { __read } from "tslib";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { AUTO_STYLE, style } from '@angular/animations';
import { getOrSetAsInMap } from '../render/shared';
import { copyObj, extractStyleParams, iteratorToArray, NG_ANIMATING_SELECTOR, NG_TRIGGER_SELECTOR, normalizeAnimationEntry, resolveTiming, SUBSTITUTION_EXPR_START, validateStyleParams, visitDslNode } from '../util';
import { parseTransitionExpr } from './animation_transition_expr';
var SELF_TOKEN = ':self';
var SELF_TOKEN_REGEX = new RegExp("s*" + SELF_TOKEN + "s*,?", 'g');
/*
 * [Validation]
 * The visitor code below will traverse the animation AST generated by the animation verb functions
 * (the output is a tree of objects) and attempt to perform a series of validations on the data. The
 * following corner-cases will be validated:
 *
 * 1. Overlap of animations
 * Given that a CSS property cannot be animated in more than one place at the same time, it's
 * important that this behavior is detected and validated. The way in which this occurs is that
 * each time a style property is examined, a string-map containing the property will be updated with
 * the start and end times for when the property is used within an animation step.
 *
 * If there are two or more parallel animations that are currently running (these are invoked by the
 * group()) on the same element then the validator will throw an error. Since the start/end timing
 * values are collected for each property then if the current animation step is animating the same
 * property and its timing values fall anywhere into the window of time that the property is
 * currently being animated within then this is what causes an error.
 *
 * 2. Timing values
 * The validator will validate to see if a timing value of `duration delay easing` or
 * `durationNumber` is valid or not.
 *
 * (note that upon validation the code below will replace the timing data with an object containing
 * {duration,delay,easing}.
 *
 * 3. Offset Validation
 * Each of the style() calls are allowed to have an offset value when placed inside of keyframes().
 * Offsets within keyframes() are considered valid when:
 *
 *   - No offsets are used at all
 *   - Each style() entry contains an offset value
 *   - Each offset is between 0 and 1
 *   - Each offset is greater to or equal than the previous one
 *
 * Otherwise an error will be thrown.
 */
export function buildAnimationAst(driver, metadata, errors) {
    return new AnimationAstBuilderVisitor(driver).build(metadata, errors);
}
var ROOT_SELECTOR = '';
var AnimationAstBuilderVisitor = /** @class */ (function () {
    function AnimationAstBuilderVisitor(_driver) {
        this._driver = _driver;
    }
    AnimationAstBuilderVisitor.prototype.build = function (metadata, errors) {
        var context = new AnimationAstBuilderContext(errors);
        this._resetContextStyleTimingState(context);
        return visitDslNode(this, normalizeAnimationEntry(metadata), context);
    };
    AnimationAstBuilderVisitor.prototype._resetContextStyleTimingState = function (context) {
        context.currentQuerySelector = ROOT_SELECTOR;
        context.collectedStyles = {};
        context.collectedStyles[ROOT_SELECTOR] = {};
        context.currentTime = 0;
    };
    AnimationAstBuilderVisitor.prototype.visitTrigger = function (metadata, context) {
        var _this = this;
        var queryCount = context.queryCount = 0;
        var depCount = context.depCount = 0;
        var states = [];
        var transitions = [];
        if (metadata.name.charAt(0) == '@') {
            context.errors.push('animation triggers cannot be prefixed with an `@` sign (e.g. trigger(\'@foo\', [...]))');
        }
        metadata.definitions.forEach(function (def) {
            _this._resetContextStyleTimingState(context);
            if (def.type == 0 /* State */) {
                var stateDef_1 = def;
                var name_1 = stateDef_1.name;
                name_1.toString().split(/\s*,\s*/).forEach(function (n) {
                    stateDef_1.name = n;
                    states.push(_this.visitState(stateDef_1, context));
                });
                stateDef_1.name = name_1;
            }
            else if (def.type == 1 /* Transition */) {
                var transition = _this.visitTransition(def, context);
                queryCount += transition.queryCount;
                depCount += transition.depCount;
                transitions.push(transition);
            }
            else {
                context.errors.push('only state() and transition() definitions can sit inside of a trigger()');
            }
        });
        return {
            type: 7 /* Trigger */,
            name: metadata.name,
            states: states,
            transitions: transitions,
            queryCount: queryCount,
            depCount: depCount,
            options: null
        };
    };
    AnimationAstBuilderVisitor.prototype.visitState = function (metadata, context) {
        var styleAst = this.visitStyle(metadata.styles, context);
        var astParams = (metadata.options && metadata.options.params) || null;
        if (styleAst.containsDynamicStyles) {
            var missingSubs_1 = new Set();
            var params_1 = astParams || {};
            styleAst.styles.forEach(function (value) {
                if (isObject(value)) {
                    var stylesObj_1 = value;
                    Object.keys(stylesObj_1).forEach(function (prop) {
                        extractStyleParams(stylesObj_1[prop]).forEach(function (sub) {
                            if (!params_1.hasOwnProperty(sub)) {
                                missingSubs_1.add(sub);
                            }
                        });
                    });
                }
            });
            if (missingSubs_1.size) {
                var missingSubsArr = iteratorToArray(missingSubs_1.values());
                context.errors.push("state(\"" + metadata
                    .name + "\", ...) must define default values for all the following style substitutions: " + missingSubsArr.join(', '));
            }
        }
        return {
            type: 0 /* State */,
            name: metadata.name,
            style: styleAst,
            options: astParams ? { params: astParams } : null
        };
    };
    AnimationAstBuilderVisitor.prototype.visitTransition = function (metadata, context) {
        context.queryCount = 0;
        context.depCount = 0;
        var animation = visitDslNode(this, normalizeAnimationEntry(metadata.animation), context);
        var matchers = parseTransitionExpr(metadata.expr, context.errors);
        return {
            type: 1 /* Transition */,
            matchers: matchers,
            animation: animation,
            queryCount: context.queryCount,
            depCount: context.depCount,
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitSequence = function (metadata, context) {
        var _this = this;
        return {
            type: 2 /* Sequence */,
            steps: metadata.steps.map(function (s) { return visitDslNode(_this, s, context); }),
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitGroup = function (metadata, context) {
        var _this = this;
        var currentTime = context.currentTime;
        var furthestTime = 0;
        var steps = metadata.steps.map(function (step) {
            context.currentTime = currentTime;
            var innerAst = visitDslNode(_this, step, context);
            furthestTime = Math.max(furthestTime, context.currentTime);
            return innerAst;
        });
        context.currentTime = furthestTime;
        return {
            type: 3 /* Group */,
            steps: steps,
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitAnimate = function (metadata, context) {
        var timingAst = constructTimingAst(metadata.timings, context.errors);
        context.currentAnimateTimings = timingAst;
        var styleAst;
        var styleMetadata = metadata.styles ? metadata.styles : style({});
        if (styleMetadata.type == 5 /* Keyframes */) {
            styleAst = this.visitKeyframes(styleMetadata, context);
        }
        else {
            var styleMetadata_1 = metadata.styles;
            var isEmpty = false;
            if (!styleMetadata_1) {
                isEmpty = true;
                var newStyleData = {};
                if (timingAst.easing) {
                    newStyleData['easing'] = timingAst.easing;
                }
                styleMetadata_1 = style(newStyleData);
            }
            context.currentTime += timingAst.duration + timingAst.delay;
            var _styleAst = this.visitStyle(styleMetadata_1, context);
            _styleAst.isEmptyStep = isEmpty;
            styleAst = _styleAst;
        }
        context.currentAnimateTimings = null;
        return {
            type: 4 /* Animate */,
            timings: timingAst,
            style: styleAst,
            options: null
        };
    };
    AnimationAstBuilderVisitor.prototype.visitStyle = function (metadata, context) {
        var ast = this._makeStyleAst(metadata, context);
        this._validateStyleAst(ast, context);
        return ast;
    };
    AnimationAstBuilderVisitor.prototype._makeStyleAst = function (metadata, context) {
        var styles = [];
        if (Array.isArray(metadata.styles)) {
            metadata.styles.forEach(function (styleTuple) {
                if (typeof styleTuple == 'string') {
                    if (styleTuple == AUTO_STYLE) {
                        styles.push(styleTuple);
                    }
                    else {
                        context.errors.push("The provided style string value " + styleTuple + " is not allowed.");
                    }
                }
                else {
                    styles.push(styleTuple);
                }
            });
        }
        else {
            styles.push(metadata.styles);
        }
        var containsDynamicStyles = false;
        var collectedEasing = null;
        styles.forEach(function (styleData) {
            if (isObject(styleData)) {
                var styleMap = styleData;
                var easing = styleMap['easing'];
                if (easing) {
                    collectedEasing = easing;
                    delete styleMap['easing'];
                }
                if (!containsDynamicStyles) {
                    for (var prop in styleMap) {
                        var value = styleMap[prop];
                        if (value.toString().indexOf(SUBSTITUTION_EXPR_START) >= 0) {
                            containsDynamicStyles = true;
                            break;
                        }
                    }
                }
            }
        });
        return {
            type: 6 /* Style */,
            styles: styles,
            easing: collectedEasing,
            offset: metadata.offset,
            containsDynamicStyles: containsDynamicStyles,
            options: null
        };
    };
    AnimationAstBuilderVisitor.prototype._validateStyleAst = function (ast, context) {
        var _this = this;
        var timings = context.currentAnimateTimings;
        var endTime = context.currentTime;
        var startTime = context.currentTime;
        if (timings && startTime > 0) {
            startTime -= timings.duration + timings.delay;
        }
        ast.styles.forEach(function (tuple) {
            if (typeof tuple == 'string')
                return;
            Object.keys(tuple).forEach(function (prop) {
                if (!_this._driver.validateStyleProperty(prop)) {
                    context.errors.push("The provided animation property \"" + prop + "\" is not a supported CSS property for animations");
                    return;
                }
                var collectedStyles = context.collectedStyles[context.currentQuerySelector];
                var collectedEntry = collectedStyles[prop];
                var updateCollectedStyle = true;
                if (collectedEntry) {
                    if (startTime != endTime && startTime >= collectedEntry.startTime &&
                        endTime <= collectedEntry.endTime) {
                        context.errors.push("The CSS property \"" + prop + "\" that exists between the times of \"" + collectedEntry.startTime + "ms\" and \"" + collectedEntry
                            .endTime + "ms\" is also being animated in a parallel animation between the times of \"" + startTime + "ms\" and \"" + endTime + "ms\"");
                        updateCollectedStyle = false;
                    }
                    // we always choose the smaller start time value since we
                    // want to have a record of the entire animation window where
                    // the style property is being animated in between
                    startTime = collectedEntry.startTime;
                }
                if (updateCollectedStyle) {
                    collectedStyles[prop] = { startTime: startTime, endTime: endTime };
                }
                if (context.options) {
                    validateStyleParams(tuple[prop], context.options, context.errors);
                }
            });
        });
    };
    AnimationAstBuilderVisitor.prototype.visitKeyframes = function (metadata, context) {
        var _this = this;
        var ast = { type: 5 /* Keyframes */, styles: [], options: null };
        if (!context.currentAnimateTimings) {
            context.errors.push("keyframes() must be placed inside of a call to animate()");
            return ast;
        }
        var MAX_KEYFRAME_OFFSET = 1;
        var totalKeyframesWithOffsets = 0;
        var offsets = [];
        var offsetsOutOfOrder = false;
        var keyframesOutOfRange = false;
        var previousOffset = 0;
        var keyframes = metadata.steps.map(function (styles) {
            var style = _this._makeStyleAst(styles, context);
            var offsetVal = style.offset != null ? style.offset : consumeOffset(style.styles);
            var offset = 0;
            if (offsetVal != null) {
                totalKeyframesWithOffsets++;
                offset = style.offset = offsetVal;
            }
            keyframesOutOfRange = keyframesOutOfRange || offset < 0 || offset > 1;
            offsetsOutOfOrder = offsetsOutOfOrder || offset < previousOffset;
            previousOffset = offset;
            offsets.push(offset);
            return style;
        });
        if (keyframesOutOfRange) {
            context.errors.push("Please ensure that all keyframe offsets are between 0 and 1");
        }
        if (offsetsOutOfOrder) {
            context.errors.push("Please ensure that all keyframe offsets are in order");
        }
        var length = metadata.steps.length;
        var generatedOffset = 0;
        if (totalKeyframesWithOffsets > 0 && totalKeyframesWithOffsets < length) {
            context.errors.push("Not all style() steps within the declared keyframes() contain offsets");
        }
        else if (totalKeyframesWithOffsets == 0) {
            generatedOffset = MAX_KEYFRAME_OFFSET / (length - 1);
        }
        var limit = length - 1;
        var currentTime = context.currentTime;
        var currentAnimateTimings = context.currentAnimateTimings;
        var animateDuration = currentAnimateTimings.duration;
        keyframes.forEach(function (kf, i) {
            var offset = generatedOffset > 0 ? (i == limit ? 1 : (generatedOffset * i)) : offsets[i];
            var durationUpToThisFrame = offset * animateDuration;
            context.currentTime = currentTime + currentAnimateTimings.delay + durationUpToThisFrame;
            currentAnimateTimings.duration = durationUpToThisFrame;
            _this._validateStyleAst(kf, context);
            kf.offset = offset;
            ast.styles.push(kf);
        });
        return ast;
    };
    AnimationAstBuilderVisitor.prototype.visitReference = function (metadata, context) {
        return {
            type: 8 /* Reference */,
            animation: visitDslNode(this, normalizeAnimationEntry(metadata.animation), context),
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitAnimateChild = function (metadata, context) {
        context.depCount++;
        return {
            type: 9 /* AnimateChild */,
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitAnimateRef = function (metadata, context) {
        return {
            type: 10 /* AnimateRef */,
            animation: this.visitReference(metadata.animation, context),
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitQuery = function (metadata, context) {
        var parentSelector = context.currentQuerySelector;
        var options = (metadata.options || {});
        context.queryCount++;
        context.currentQuery = metadata;
        var _a = __read(normalizeSelector(metadata.selector), 2), selector = _a[0], includeSelf = _a[1];
        context.currentQuerySelector =
            parentSelector.length ? (parentSelector + ' ' + selector) : selector;
        getOrSetAsInMap(context.collectedStyles, context.currentQuerySelector, {});
        var animation = visitDslNode(this, normalizeAnimationEntry(metadata.animation), context);
        context.currentQuery = null;
        context.currentQuerySelector = parentSelector;
        return {
            type: 11 /* Query */,
            selector: selector,
            limit: options.limit || 0,
            optional: !!options.optional,
            includeSelf: includeSelf,
            animation: animation,
            originalSelector: metadata.selector,
            options: normalizeAnimationOptions(metadata.options)
        };
    };
    AnimationAstBuilderVisitor.prototype.visitStagger = function (metadata, context) {
        if (!context.currentQuery) {
            context.errors.push("stagger() can only be used inside of query()");
        }
        var timings = metadata.timings === 'full' ?
            { duration: 0, delay: 0, easing: 'full' } :
            resolveTiming(metadata.timings, context.errors, true);
        return {
            type: 12 /* Stagger */,
            animation: visitDslNode(this, normalizeAnimationEntry(metadata.animation), context),
            timings: timings,
            options: null
        };
    };
    return AnimationAstBuilderVisitor;
}());
export { AnimationAstBuilderVisitor };
function normalizeSelector(selector) {
    var hasAmpersand = selector.split(/\s*,\s*/).find(function (token) { return token == SELF_TOKEN; }) ? true : false;
    if (hasAmpersand) {
        selector = selector.replace(SELF_TOKEN_REGEX, '');
    }
    // the :enter and :leave selectors are filled in at runtime during timeline building
    selector = selector.replace(/@\*/g, NG_TRIGGER_SELECTOR)
        .replace(/@\w+/g, function (match) { return NG_TRIGGER_SELECTOR + '-' + match.substr(1); })
        .replace(/:animating/g, NG_ANIMATING_SELECTOR);
    return [selector, hasAmpersand];
}
function normalizeParams(obj) {
    return obj ? copyObj(obj) : null;
}
var AnimationAstBuilderContext = /** @class */ (function () {
    function AnimationAstBuilderContext(errors) {
        this.errors = errors;
        this.queryCount = 0;
        this.depCount = 0;
        this.currentTransition = null;
        this.currentQuery = null;
        this.currentQuerySelector = null;
        this.currentAnimateTimings = null;
        this.currentTime = 0;
        this.collectedStyles = {};
        this.options = null;
    }
    return AnimationAstBuilderContext;
}());
export { AnimationAstBuilderContext };
function consumeOffset(styles) {
    if (typeof styles == 'string')
        return null;
    var offset = null;
    if (Array.isArray(styles)) {
        styles.forEach(function (styleTuple) {
            if (isObject(styleTuple) && styleTuple.hasOwnProperty('offset')) {
                var obj = styleTuple;
                offset = parseFloat(obj['offset']);
                delete obj['offset'];
            }
        });
    }
    else if (isObject(styles) && styles.hasOwnProperty('offset')) {
        var obj = styles;
        offset = parseFloat(obj['offset']);
        delete obj['offset'];
    }
    return offset;
}
function isObject(value) {
    return !Array.isArray(value) && typeof value == 'object';
}
function constructTimingAst(value, errors) {
    var timings = null;
    if (value.hasOwnProperty('duration')) {
        timings = value;
    }
    else if (typeof value == 'number') {
        var duration = resolveTiming(value, errors).duration;
        return makeTimingAst(duration, 0, '');
    }
    var strValue = value;
    var isDynamic = strValue.split(/\s+/).some(function (v) { return v.charAt(0) == '{' && v.charAt(1) == '{'; });
    if (isDynamic) {
        var ast = makeTimingAst(0, 0, '');
        ast.dynamic = true;
        ast.strValue = strValue;
        return ast;
    }
    timings = timings || resolveTiming(strValue, errors);
    return makeTimingAst(timings.duration, timings.delay, timings.easing);
}
function normalizeAnimationOptions(options) {
    if (options) {
        options = copyObj(options);
        if (options['params']) {
            options['params'] = normalizeParams(options['params']);
        }
    }
    else {
        options = {};
    }
    return options;
}
function makeTimingAst(duration, delay, easing) {
    return { duration: duration, delay: delay, easing: easing };
}
//# sourceMappingURL=data:application/json;base64,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