/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** IDs are delimited by an empty space, as per the spec. */
var ID_DELIMITER = ' ';
/**
 * Adds the given ID to the specified ARIA attribute on an element.
 * Used for attributes such as aria-labelledby, aria-owns, etc.
 */
export function addAriaReferencedId(el, attr, id) {
    var ids = getAriaReferenceIds(el, attr);
    if (ids.some(function (existingId) { return existingId.trim() == id.trim(); })) {
        return;
    }
    ids.push(id.trim());
    el.setAttribute(attr, ids.join(ID_DELIMITER));
}
/**
 * Removes the given ID from the specified ARIA attribute on an element.
 * Used for attributes such as aria-labelledby, aria-owns, etc.
 */
export function removeAriaReferencedId(el, attr, id) {
    var ids = getAriaReferenceIds(el, attr);
    var filteredIds = ids.filter(function (val) { return val != id.trim(); });
    if (filteredIds.length) {
        el.setAttribute(attr, filteredIds.join(ID_DELIMITER));
    }
    else {
        el.removeAttribute(attr);
    }
}
/**
 * Gets the list of IDs referenced by the given ARIA attribute on an element.
 * Used for attributes such as aria-labelledby, aria-owns, etc.
 */
export function getAriaReferenceIds(el, attr) {
    // Get string array of all individual ids (whitespace delimited) in the attribute value
    return (el.getAttribute(attr) || '').match(/\S+/g) || [];
}
//# sourceMappingURL=data:application/json;base64,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