/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as i18n from '../i18n_ast';
export class Serializer {
    // Creates a name mapper, see `PlaceholderMapper`
    // Returning `null` means that no name mapping is used.
    createNameMapper(message) {
        return null;
    }
}
/**
 * A simple mapper that take a function to transform an internal name to a public name
 */
export class SimplePlaceholderMapper extends i18n.RecurseVisitor {
    // create a mapping from the message
    constructor(message, mapName) {
        super();
        this.mapName = mapName;
        this.internalToPublic = {};
        this.publicToNextId = {};
        this.publicToInternal = {};
        message.nodes.forEach(node => node.visit(this));
    }
    toPublicName(internalName) {
        return this.internalToPublic.hasOwnProperty(internalName) ?
            this.internalToPublic[internalName] :
            null;
    }
    toInternalName(publicName) {
        return this.publicToInternal.hasOwnProperty(publicName) ? this.publicToInternal[publicName] :
            null;
    }
    visitText(text, context) {
        return null;
    }
    visitTagPlaceholder(ph, context) {
        this.visitPlaceholderName(ph.startName);
        super.visitTagPlaceholder(ph, context);
        this.visitPlaceholderName(ph.closeName);
    }
    visitPlaceholder(ph, context) {
        this.visitPlaceholderName(ph.name);
    }
    visitIcuPlaceholder(ph, context) {
        this.visitPlaceholderName(ph.name);
    }
    // XMB placeholders could only contains A-Z, 0-9 and _
    visitPlaceholderName(internalName) {
        if (!internalName || this.internalToPublic.hasOwnProperty(internalName)) {
            return;
        }
        let publicName = this.mapName(internalName);
        if (this.publicToInternal.hasOwnProperty(publicName)) {
            // Create a new XMB when it has already been used
            const nextId = this.publicToNextId[publicName];
            this.publicToNextId[publicName] = nextId + 1;
            publicName = `${publicName}_${nextId}`;
        }
        else {
            this.publicToNextId[publicName] = 1;
        }
        this.internalToPublic[internalName] = publicName;
        this.publicToInternal[publicName] = internalName;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VyaWFsaXplci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvbXBpbGVyL3NyYy9pMThuL3NlcmlhbGl6ZXJzL3NlcmlhbGl6ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBRUgsT0FBTyxLQUFLLElBQUksTUFBTSxhQUFhLENBQUM7QUFFcEMsTUFBTSxPQUFnQixVQUFVO0lBVzlCLGlEQUFpRDtJQUNqRCx1REFBdUQ7SUFDdkQsZ0JBQWdCLENBQUMsT0FBcUI7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0NBQ0Y7QUFjRDs7R0FFRztBQUNILE1BQU0sT0FBTyx1QkFBd0IsU0FBUSxJQUFJLENBQUMsY0FBYztJQUs5RCxvQ0FBb0M7SUFDcEMsWUFBWSxPQUFxQixFQUFVLE9BQWlDO1FBQzFFLEtBQUssRUFBRSxDQUFDO1FBRGlDLFlBQU8sR0FBUCxPQUFPLENBQTBCO1FBTHBFLHFCQUFnQixHQUEwQixFQUFFLENBQUM7UUFDN0MsbUJBQWMsR0FBMEIsRUFBRSxDQUFDO1FBQzNDLHFCQUFnQixHQUEwQixFQUFFLENBQUM7UUFLbkQsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVELFlBQVksQ0FBQyxZQUFvQjtRQUMvQixPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUN2RCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNyQyxJQUFJLENBQUM7SUFDWCxDQUFDO0lBRUQsY0FBYyxDQUFDLFVBQWtCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDbkMsSUFBSSxDQUFDO0lBQ2pFLENBQUM7SUFFRCxTQUFTLENBQUMsSUFBZSxFQUFFLE9BQWE7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsbUJBQW1CLENBQUMsRUFBdUIsRUFBRSxPQUFhO1FBQ3hELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDeEMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRCxnQkFBZ0IsQ0FBQyxFQUFvQixFQUFFLE9BQWE7UUFDbEQsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQsbUJBQW1CLENBQUMsRUFBdUIsRUFBRSxPQUFhO1FBQ3hELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVELHNEQUFzRDtJQUM5QyxvQkFBb0IsQ0FBQyxZQUFvQjtRQUMvQyxJQUFJLENBQUMsWUFBWSxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDdkUsT0FBTztTQUNSO1FBRUQsSUFBSSxVQUFVLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUU1QyxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDcEQsaURBQWlEO1lBQ2pELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDL0MsSUFBSSxDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUMsR0FBRyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1lBQzdDLFVBQVUsR0FBRyxHQUFHLFVBQVUsSUFBSSxNQUFNLEVBQUUsQ0FBQztTQUN4QzthQUFNO1lBQ0wsSUFBSSxDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDckM7UUFFRCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLEdBQUcsVUFBVSxDQUFDO1FBQ2pELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsR0FBRyxZQUFZLENBQUM7SUFDbkQsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBJbmMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQgKiBhcyBpMThuIGZyb20gJy4uL2kxOG5fYXN0JztcblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFNlcmlhbGl6ZXIge1xuICAvLyAtIFRoZSBgcGxhY2Vob2xkZXJzYCBhbmQgYHBsYWNlaG9sZGVyVG9NZXNzYWdlYCBwcm9wZXJ0aWVzIGFyZSBpcnJlbGV2YW50IGluIHRoZSBpbnB1dCBtZXNzYWdlc1xuICAvLyAtIFRoZSBgaWRgIGNvbnRhaW5zIHRoZSBtZXNzYWdlIGlkIHRoYXQgdGhlIHNlcmlhbGl6ZXIgaXMgZXhwZWN0ZWQgdG8gdXNlXG4gIC8vIC0gUGxhY2Vob2xkZXIgbmFtZXMgYXJlIGFscmVhZHkgbWFwIHRvIHB1YmxpYyBuYW1lcyB1c2luZyB0aGUgcHJvdmlkZWQgbWFwcGVyXG4gIGFic3RyYWN0IHdyaXRlKG1lc3NhZ2VzOiBpMThuLk1lc3NhZ2VbXSwgbG9jYWxlOiBzdHJpbmd8bnVsbCk6IHN0cmluZztcblxuICBhYnN0cmFjdCBsb2FkKGNvbnRlbnQ6IHN0cmluZywgdXJsOiBzdHJpbmcpOlxuICAgICAge2xvY2FsZTogc3RyaW5nfG51bGwsIGkxOG5Ob2Rlc0J5TXNnSWQ6IHtbbXNnSWQ6IHN0cmluZ106IGkxOG4uTm9kZVtdfX07XG5cbiAgYWJzdHJhY3QgZGlnZXN0KG1lc3NhZ2U6IGkxOG4uTWVzc2FnZSk6IHN0cmluZztcblxuICAvLyBDcmVhdGVzIGEgbmFtZSBtYXBwZXIsIHNlZSBgUGxhY2Vob2xkZXJNYXBwZXJgXG4gIC8vIFJldHVybmluZyBgbnVsbGAgbWVhbnMgdGhhdCBubyBuYW1lIG1hcHBpbmcgaXMgdXNlZC5cbiAgY3JlYXRlTmFtZU1hcHBlcihtZXNzYWdlOiBpMThuLk1lc3NhZ2UpOiBQbGFjZWhvbGRlck1hcHBlcnxudWxsIHtcbiAgICByZXR1cm4gbnVsbDtcbiAgfVxufVxuXG4vKipcbiAqIEEgYFBsYWNlaG9sZGVyTWFwcGVyYCBjb252ZXJ0cyBwbGFjZWhvbGRlciBuYW1lcyBmcm9tIGludGVybmFsIHRvIHNlcmlhbGl6ZWQgcmVwcmVzZW50YXRpb24gYW5kXG4gKiBiYWNrLlxuICpcbiAqIEl0IHNob3VsZCBiZSB1c2VkIGZvciBzZXJpYWxpemF0aW9uIGZvcm1hdCB0aGF0IHB1dCBjb25zdHJhaW50cyBvbiB0aGUgcGxhY2Vob2xkZXIgbmFtZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUGxhY2Vob2xkZXJNYXBwZXIge1xuICB0b1B1YmxpY05hbWUoaW50ZXJuYWxOYW1lOiBzdHJpbmcpOiBzdHJpbmd8bnVsbDtcblxuICB0b0ludGVybmFsTmFtZShwdWJsaWNOYW1lOiBzdHJpbmcpOiBzdHJpbmd8bnVsbDtcbn1cblxuLyoqXG4gKiBBIHNpbXBsZSBtYXBwZXIgdGhhdCB0YWtlIGEgZnVuY3Rpb24gdG8gdHJhbnNmb3JtIGFuIGludGVybmFsIG5hbWUgdG8gYSBwdWJsaWMgbmFtZVxuICovXG5leHBvcnQgY2xhc3MgU2ltcGxlUGxhY2Vob2xkZXJNYXBwZXIgZXh0ZW5kcyBpMThuLlJlY3Vyc2VWaXNpdG9yIGltcGxlbWVudHMgUGxhY2Vob2xkZXJNYXBwZXIge1xuICBwcml2YXRlIGludGVybmFsVG9QdWJsaWM6IHtbazogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuICBwcml2YXRlIHB1YmxpY1RvTmV4dElkOiB7W2s6IHN0cmluZ106IG51bWJlcn0gPSB7fTtcbiAgcHJpdmF0ZSBwdWJsaWNUb0ludGVybmFsOiB7W2s6IHN0cmluZ106IHN0cmluZ30gPSB7fTtcblxuICAvLyBjcmVhdGUgYSBtYXBwaW5nIGZyb20gdGhlIG1lc3NhZ2VcbiAgY29uc3RydWN0b3IobWVzc2FnZTogaTE4bi5NZXNzYWdlLCBwcml2YXRlIG1hcE5hbWU6IChuYW1lOiBzdHJpbmcpID0+IHN0cmluZykge1xuICAgIHN1cGVyKCk7XG4gICAgbWVzc2FnZS5ub2Rlcy5mb3JFYWNoKG5vZGUgPT4gbm9kZS52aXNpdCh0aGlzKSk7XG4gIH1cblxuICB0b1B1YmxpY05hbWUoaW50ZXJuYWxOYW1lOiBzdHJpbmcpOiBzdHJpbmd8bnVsbCB7XG4gICAgcmV0dXJuIHRoaXMuaW50ZXJuYWxUb1B1YmxpYy5oYXNPd25Qcm9wZXJ0eShpbnRlcm5hbE5hbWUpID9cbiAgICAgICAgdGhpcy5pbnRlcm5hbFRvUHVibGljW2ludGVybmFsTmFtZV0gOlxuICAgICAgICBudWxsO1xuICB9XG5cbiAgdG9JbnRlcm5hbE5hbWUocHVibGljTmFtZTogc3RyaW5nKTogc3RyaW5nfG51bGwge1xuICAgIHJldHVybiB0aGlzLnB1YmxpY1RvSW50ZXJuYWwuaGFzT3duUHJvcGVydHkocHVibGljTmFtZSkgPyB0aGlzLnB1YmxpY1RvSW50ZXJuYWxbcHVibGljTmFtZV0gOlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBudWxsO1xuICB9XG5cbiAgdmlzaXRUZXh0KHRleHQ6IGkxOG4uVGV4dCwgY29udGV4dD86IGFueSk6IGFueSB7XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cblxuICB2aXNpdFRhZ1BsYWNlaG9sZGVyKHBoOiBpMThuLlRhZ1BsYWNlaG9sZGVyLCBjb250ZXh0PzogYW55KTogYW55IHtcbiAgICB0aGlzLnZpc2l0UGxhY2Vob2xkZXJOYW1lKHBoLnN0YXJ0TmFtZSk7XG4gICAgc3VwZXIudmlzaXRUYWdQbGFjZWhvbGRlcihwaCwgY29udGV4dCk7XG4gICAgdGhpcy52aXNpdFBsYWNlaG9sZGVyTmFtZShwaC5jbG9zZU5hbWUpO1xuICB9XG5cbiAgdmlzaXRQbGFjZWhvbGRlcihwaDogaTE4bi5QbGFjZWhvbGRlciwgY29udGV4dD86IGFueSk6IGFueSB7XG4gICAgdGhpcy52aXNpdFBsYWNlaG9sZGVyTmFtZShwaC5uYW1lKTtcbiAgfVxuXG4gIHZpc2l0SWN1UGxhY2Vob2xkZXIocGg6IGkxOG4uSWN1UGxhY2Vob2xkZXIsIGNvbnRleHQ/OiBhbnkpOiBhbnkge1xuICAgIHRoaXMudmlzaXRQbGFjZWhvbGRlck5hbWUocGgubmFtZSk7XG4gIH1cblxuICAvLyBYTUIgcGxhY2Vob2xkZXJzIGNvdWxkIG9ubHkgY29udGFpbnMgQS1aLCAwLTkgYW5kIF9cbiAgcHJpdmF0ZSB2aXNpdFBsYWNlaG9sZGVyTmFtZShpbnRlcm5hbE5hbWU6IHN0cmluZyk6IHZvaWQge1xuICAgIGlmICghaW50ZXJuYWxOYW1lIHx8IHRoaXMuaW50ZXJuYWxUb1B1YmxpYy5oYXNPd25Qcm9wZXJ0eShpbnRlcm5hbE5hbWUpKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgbGV0IHB1YmxpY05hbWUgPSB0aGlzLm1hcE5hbWUoaW50ZXJuYWxOYW1lKTtcblxuICAgIGlmICh0aGlzLnB1YmxpY1RvSW50ZXJuYWwuaGFzT3duUHJvcGVydHkocHVibGljTmFtZSkpIHtcbiAgICAgIC8vIENyZWF0ZSBhIG5ldyBYTUIgd2hlbiBpdCBoYXMgYWxyZWFkeSBiZWVuIHVzZWRcbiAgICAgIGNvbnN0IG5leHRJZCA9IHRoaXMucHVibGljVG9OZXh0SWRbcHVibGljTmFtZV07XG4gICAgICB0aGlzLnB1YmxpY1RvTmV4dElkW3B1YmxpY05hbWVdID0gbmV4dElkICsgMTtcbiAgICAgIHB1YmxpY05hbWUgPSBgJHtwdWJsaWNOYW1lfV8ke25leHRJZH1gO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnB1YmxpY1RvTmV4dElkW3B1YmxpY05hbWVdID0gMTtcbiAgICB9XG5cbiAgICB0aGlzLmludGVybmFsVG9QdWJsaWNbaW50ZXJuYWxOYW1lXSA9IHB1YmxpY05hbWU7XG4gICAgdGhpcy5wdWJsaWNUb0ludGVybmFsW3B1YmxpY05hbWVdID0gaW50ZXJuYWxOYW1lO1xuICB9XG59XG4iXX0=