import { __read, __spread } from "tslib";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as i18n from '../../../i18n/i18n_ast';
import { toPublicName } from '../../../i18n/serializers/xmb';
import * as o from '../../../output/output_ast';
/* Closure variables holding messages must be named `MSG_[A-Z0-9]+` */
var CLOSURE_TRANSLATION_PREFIX = 'MSG_';
/* Prefix for non-`goog.getMsg` i18n-related vars */
export var TRANSLATION_PREFIX = 'I18N_';
/** Name of the i18n attributes **/
export var I18N_ATTR = 'i18n';
export var I18N_ATTR_PREFIX = 'i18n-';
/** Prefix of var expressions used in ICUs */
export var I18N_ICU_VAR_PREFIX = 'VAR_';
/** Prefix of ICU expressions for post processing */
export var I18N_ICU_MAPPING_PREFIX = 'I18N_EXP_';
/** Placeholder wrapper for i18n expressions **/
export var I18N_PLACEHOLDER_SYMBOL = '�';
export function isI18nAttribute(name) {
    return name === I18N_ATTR || name.startsWith(I18N_ATTR_PREFIX);
}
export function isI18nRootNode(meta) {
    return meta instanceof i18n.Message;
}
export function isSingleI18nIcu(meta) {
    return isI18nRootNode(meta) && meta.nodes.length === 1 && meta.nodes[0] instanceof i18n.Icu;
}
export function hasI18nMeta(node) {
    return !!node.i18n;
}
export function hasI18nAttrs(element) {
    return element.attrs.some(function (attr) { return isI18nAttribute(attr.name); });
}
export function icuFromI18nMessage(message) {
    return message.nodes[0];
}
export function wrapI18nPlaceholder(content, contextId) {
    if (contextId === void 0) { contextId = 0; }
    var blockId = contextId > 0 ? ":" + contextId : '';
    return "" + I18N_PLACEHOLDER_SYMBOL + content + blockId + I18N_PLACEHOLDER_SYMBOL;
}
export function assembleI18nBoundString(strings, bindingStartIndex, contextId) {
    if (bindingStartIndex === void 0) { bindingStartIndex = 0; }
    if (contextId === void 0) { contextId = 0; }
    if (!strings.length)
        return '';
    var acc = '';
    var lastIdx = strings.length - 1;
    for (var i = 0; i < lastIdx; i++) {
        acc += "" + strings[i] + wrapI18nPlaceholder(bindingStartIndex + i, contextId);
    }
    acc += strings[lastIdx];
    return acc;
}
export function getSeqNumberGenerator(startsAt) {
    if (startsAt === void 0) { startsAt = 0; }
    var current = startsAt;
    return function () { return current++; };
}
export function placeholdersToParams(placeholders) {
    var params = {};
    placeholders.forEach(function (values, key) {
        params[key] = o.literal(values.length > 1 ? "[" + values.join('|') + "]" : values[0]);
    });
    return params;
}
export function updatePlaceholderMap(map, name) {
    var values = [];
    for (var _i = 2; _i < arguments.length; _i++) {
        values[_i - 2] = arguments[_i];
    }
    var current = map.get(name) || [];
    current.push.apply(current, __spread(values));
    map.set(name, current);
}
export function assembleBoundTextPlaceholders(meta, bindingStartIndex, contextId) {
    if (bindingStartIndex === void 0) { bindingStartIndex = 0; }
    if (contextId === void 0) { contextId = 0; }
    var startIdx = bindingStartIndex;
    var placeholders = new Map();
    var node = meta instanceof i18n.Message ? meta.nodes.find(function (node) { return node instanceof i18n.Container; }) : meta;
    if (node) {
        node
            .children
            .filter(function (child) { return child instanceof i18n.Placeholder; })
            .forEach(function (child, idx) {
            var content = wrapI18nPlaceholder(startIdx + idx, contextId);
            updatePlaceholderMap(placeholders, child.name, content);
        });
    }
    return placeholders;
}
/**
 * Format the placeholder names in a map of placeholders to expressions.
 *
 * The placeholder names are converted from "internal" format (e.g. `START_TAG_DIV_1`) to "external"
 * format (e.g. `startTagDiv_1`).
 *
 * @param params A map of placeholder names to expressions.
 * @param useCamelCase whether to camelCase the placeholder name when formatting.
 * @returns A new map of formatted placeholder names to expressions.
 */
export function i18nFormatPlaceholderNames(params, useCamelCase) {
    if (params === void 0) { params = {}; }
    var _params = {};
    if (params && Object.keys(params).length) {
        Object.keys(params).forEach(function (key) { return _params[formatI18nPlaceholderName(key, useCamelCase)] = params[key]; });
    }
    return _params;
}
/**
 * Converts internal placeholder names to public-facing format
 * (for example to use in goog.getMsg call).
 * Example: `START_TAG_DIV_1` is converted to `startTagDiv_1`.
 *
 * @param name The placeholder name that should be formatted
 * @returns Formatted placeholder name
 */
export function formatI18nPlaceholderName(name, useCamelCase) {
    if (useCamelCase === void 0) { useCamelCase = true; }
    var publicName = toPublicName(name);
    if (!useCamelCase) {
        return publicName;
    }
    var chunks = publicName.split('_');
    if (chunks.length === 1) {
        // if no "_" found - just lowercase the value
        return name.toLowerCase();
    }
    var postfix;
    // eject last element if it's a number
    if (/^\d+$/.test(chunks[chunks.length - 1])) {
        postfix = chunks.pop();
    }
    var raw = chunks.shift().toLowerCase();
    if (chunks.length) {
        raw += chunks.map(function (c) { return c.charAt(0).toUpperCase() + c.slice(1).toLowerCase(); }).join('');
    }
    return postfix ? raw + "_" + postfix : raw;
}
/**
 * Generates a prefix for translation const name.
 *
 * @param extra Additional local prefix that should be injected into translation var name
 * @returns Complete translation const prefix
 */
export function getTranslationConstPrefix(extra) {
    return ("" + CLOSURE_TRANSLATION_PREFIX + extra).toUpperCase();
}
/**
 * Generate AST to declare a variable. E.g. `var I18N_1;`.
 * @param variable the name of the variable to declare.
 */
export function declareI18nVariable(variable) {
    return new o.DeclareVarStmt(variable.name, undefined, o.INFERRED_TYPE, null, variable.sourceSpan);
}
//# sourceMappingURL=data:application/json;base64,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