(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/ngtsc/imports/src/emitter", ["require", "exports", "tslib", "@angular/compiler", "typescript", "@angular/compiler-cli/src/ngtsc/file_system", "@angular/compiler-cli/src/ngtsc/file_system/src/util", "@angular/compiler-cli/src/ngtsc/util/src/typescript", "@angular/compiler-cli/src/ngtsc/imports/src/find_export"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    var tslib_1 = require("tslib");
    /**
     * @license
     * Copyright Google Inc. All Rights Reserved.
     *
     * Use of this source code is governed by an MIT-style license that can be
     * found in the LICENSE file at https://angular.io/license
     */
    var compiler_1 = require("@angular/compiler");
    var ts = require("typescript");
    var file_system_1 = require("@angular/compiler-cli/src/ngtsc/file_system");
    var util_1 = require("@angular/compiler-cli/src/ngtsc/file_system/src/util");
    var typescript_1 = require("@angular/compiler-cli/src/ngtsc/util/src/typescript");
    var find_export_1 = require("@angular/compiler-cli/src/ngtsc/imports/src/find_export");
    /**
     * Flags which alter the imports generated by the `ReferenceEmitter`.
     */
    var ImportFlags;
    (function (ImportFlags) {
        ImportFlags[ImportFlags["None"] = 0] = "None";
        /**
         * Force the generation of a new import when generating a reference, even if an identifier already
         * exists in the target file which could be used instead.
         *
         * This is sometimes required if there's a risk TypeScript might remove imports during emit.
         */
        ImportFlags[ImportFlags["ForceNewImport"] = 1] = "ForceNewImport";
        /**
         * Don't make use of any aliasing information when emitting a reference.
         *
         * This is sometimes required if emitting into a context where generated references will be fed
         * into TypeScript and type-checked (such as in template type-checking).
         */
        ImportFlags[ImportFlags["NoAliasing"] = 2] = "NoAliasing";
        /**
         * Indicates that an import to a type-only declaration is allowed.
         *
         * For references that occur in type-positions, the referred declaration may be a type-only
         * declaration that is not retained during emit. Including this flag allows to emit references to
         * type-only declarations as used in e.g. template type-checking.
         */
        ImportFlags[ImportFlags["AllowTypeImports"] = 4] = "AllowTypeImports";
    })(ImportFlags = exports.ImportFlags || (exports.ImportFlags = {}));
    /**
     * Generates `Expression`s which refer to `Reference`s in a given context.
     *
     * A `ReferenceEmitter` uses one or more `ReferenceEmitStrategy` implementations to produce an
     * `Expression` which refers to a `Reference` in the context of a particular file.
     */
    var ReferenceEmitter = /** @class */ (function () {
        function ReferenceEmitter(strategies) {
            this.strategies = strategies;
        }
        ReferenceEmitter.prototype.emit = function (ref, context, importFlags) {
            var e_1, _a;
            if (importFlags === void 0) { importFlags = ImportFlags.None; }
            try {
                for (var _b = tslib_1.__values(this.strategies), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var strategy = _c.value;
                    var emitted = strategy.emit(ref, context, importFlags);
                    if (emitted !== null) {
                        return emitted;
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
            throw new Error("Unable to write a reference to " + typescript_1.nodeNameForError(ref.node) + " in " + ref.node.getSourceFile().fileName + " from " + context.fileName);
        };
        return ReferenceEmitter;
    }());
    exports.ReferenceEmitter = ReferenceEmitter;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations by any local `ts.Identifier`s, if
     * such identifiers are available.
     */
    var LocalIdentifierStrategy = /** @class */ (function () {
        function LocalIdentifierStrategy() {
        }
        LocalIdentifierStrategy.prototype.emit = function (ref, context, importFlags) {
            // If the emitter has specified ForceNewImport, then LocalIdentifierStrategy should not use a
            // local identifier at all, *except* in the source file where the node is actually declared.
            if (importFlags & ImportFlags.ForceNewImport &&
                typescript_1.getSourceFile(ref.node) !== typescript_1.getSourceFile(context)) {
                return null;
            }
            // A Reference can have multiple identities in different files, so it may already have an
            // Identifier in the requested context file.
            var identifier = ref.getIdentityIn(context);
            if (identifier !== null) {
                return new compiler_1.WrappedNodeExpr(identifier);
            }
            else {
                return null;
            }
        };
        return LocalIdentifierStrategy;
    }());
    exports.LocalIdentifierStrategy = LocalIdentifierStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations that come from `node_modules` using
     * an absolute import.
     *
     * Part of this strategy involves looking at the target entry point and identifying the exported
     * name of the targeted declaration, as it might be different from the declared name (e.g. a
     * directive might be declared as FooDirImpl, but exported as FooDir). If no export can be found
     * which maps back to the original directive, an error is thrown.
     */
    var AbsoluteModuleStrategy = /** @class */ (function () {
        function AbsoluteModuleStrategy(program, checker, moduleResolver, reflectionHost) {
            this.program = program;
            this.checker = checker;
            this.moduleResolver = moduleResolver;
            this.reflectionHost = reflectionHost;
            /**
             * A cache of the exports of specific modules, because resolving a module to its exports is a
             * costly operation.
             */
            this.moduleExportsCache = new Map();
        }
        AbsoluteModuleStrategy.prototype.emit = function (ref, context, importFlags) {
            if (ref.bestGuessOwningModule === null) {
                // There is no module name available for this Reference, meaning it was arrived at via a
                // relative path.
                return null;
            }
            else if (!typescript_1.isDeclaration(ref.node)) {
                // It's not possible to import something which isn't a declaration.
                throw new Error("Debug assert: unable to import a Reference to non-declaration of type " + ts.SyntaxKind[ref.node.kind] + ".");
            }
            else if ((importFlags & ImportFlags.AllowTypeImports) === 0 && typescript_1.isTypeDeclaration(ref.node)) {
                throw new Error("Importing a type-only declaration of type " + ts.SyntaxKind[ref.node.kind] + " in a value position is not allowed.");
            }
            // Try to find the exported name of the declaration, if one is available.
            var _a = ref.bestGuessOwningModule, specifier = _a.specifier, resolutionContext = _a.resolutionContext;
            var symbolName = this.resolveImportName(specifier, ref.node, resolutionContext);
            if (symbolName === null) {
                // TODO(alxhub): make this error a ts.Diagnostic pointing at whatever caused this import to be
                // triggered.
                throw new Error("Symbol " + ref.debugName + " declared in " + typescript_1.getSourceFile(ref.node).fileName + " is not exported from " + specifier + " (import into " + context.fileName + ")");
            }
            return new compiler_1.ExternalExpr(new compiler_1.ExternalReference(specifier, symbolName));
        };
        AbsoluteModuleStrategy.prototype.resolveImportName = function (moduleName, target, fromFile) {
            var exports = this.getExportsOfModule(moduleName, fromFile);
            if (exports !== null && exports.has(target)) {
                return exports.get(target);
            }
            else {
                return null;
            }
        };
        AbsoluteModuleStrategy.prototype.getExportsOfModule = function (moduleName, fromFile) {
            if (!this.moduleExportsCache.has(moduleName)) {
                this.moduleExportsCache.set(moduleName, this.enumerateExportsOfModule(moduleName, fromFile));
            }
            return this.moduleExportsCache.get(moduleName);
        };
        AbsoluteModuleStrategy.prototype.enumerateExportsOfModule = function (specifier, fromFile) {
            // First, resolve the module specifier to its entry point, and get the ts.Symbol for it.
            var entryPointFile = this.moduleResolver.resolveModule(specifier, fromFile);
            if (entryPointFile === null) {
                return null;
            }
            var exports = this.reflectionHost.getExportsOfModule(entryPointFile);
            if (exports === null) {
                return null;
            }
            var exportMap = new Map();
            exports.forEach(function (declaration, name) {
                // It's okay to skip inline declarations, since by definition they're not target-able with a
                // ts.Declaration anyway.
                if (declaration.node !== null) {
                    exportMap.set(declaration.node, name);
                }
            });
            return exportMap;
        };
        return AbsoluteModuleStrategy;
    }());
    exports.AbsoluteModuleStrategy = AbsoluteModuleStrategy;
    /**
     * A `ReferenceEmitStrategy` which will refer to declarations via relative paths, provided they're
     * both in the logical project "space" of paths.
     *
     * This is trickier than it sounds, as the two files may be in different root directories in the
     * project. Simply calculating a file system relative path between the two is not sufficient.
     * Instead, `LogicalProjectPath`s are used.
     */
    var LogicalProjectStrategy = /** @class */ (function () {
        function LogicalProjectStrategy(reflector, logicalFs) {
            this.reflector = reflector;
            this.logicalFs = logicalFs;
        }
        LogicalProjectStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            // Compute the relative path from the importing file to the file being imported. This is done
            // as a logical path computation, because the two files might be in different rootDirs.
            var destPath = this.logicalFs.logicalPathOfSf(destSf);
            if (destPath === null) {
                // The imported file is not within the logical project filesystem.
                return null;
            }
            var originPath = this.logicalFs.logicalPathOfSf(context);
            if (originPath === null) {
                throw new Error("Debug assert: attempt to import from " + context.fileName + " but it's outside the program?");
            }
            // There's no way to emit a relative reference from a file to itself.
            if (destPath === originPath) {
                return null;
            }
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                // The target declaration isn't exported from the file it's declared in. This is an issue!
                return null;
            }
            // With both files expressed as LogicalProjectPaths, getting the module specifier as a relative
            // path is now straightforward.
            var moduleName = file_system_1.LogicalProjectPath.relativePathBetween(originPath, destPath);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return LogicalProjectStrategy;
    }());
    exports.LogicalProjectStrategy = LogicalProjectStrategy;
    /**
     * A `ReferenceEmitStrategy` which constructs relatives paths between `ts.SourceFile`s.
     *
     * This strategy can be used if there is no `rootDir`/`rootDirs` structure for the project which
     * necessitates the stronger logic of `LogicalProjectStrategy`.
     */
    var RelativePathStrategy = /** @class */ (function () {
        function RelativePathStrategy(reflector) {
            this.reflector = reflector;
        }
        RelativePathStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var moduleName = util_1.stripExtension(file_system_1.relative(file_system_1.dirname(file_system_1.absoluteFromSourceFile(context)), file_system_1.absoluteFromSourceFile(destSf)));
            if (!moduleName.startsWith('../')) {
                moduleName = ('./' + moduleName);
            }
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return RelativePathStrategy;
    }());
    exports.RelativePathStrategy = RelativePathStrategy;
    /**
     * A `ReferenceEmitStrategy` which uses a `UnifiedModulesHost` to generate absolute import
     * references.
     */
    var UnifiedModulesStrategy = /** @class */ (function () {
        function UnifiedModulesStrategy(reflector, unifiedModulesHost) {
            this.reflector = reflector;
            this.unifiedModulesHost = unifiedModulesHost;
        }
        UnifiedModulesStrategy.prototype.emit = function (ref, context) {
            var destSf = typescript_1.getSourceFile(ref.node);
            var name = find_export_1.findExportedNameOfNode(ref.node, destSf, this.reflector);
            if (name === null) {
                return null;
            }
            var moduleName = this.unifiedModulesHost.fileNameToModuleName(destSf.fileName, context.fileName);
            return new compiler_1.ExternalExpr({ moduleName: moduleName, name: name });
        };
        return UnifiedModulesStrategy;
    }());
    exports.UnifiedModulesStrategy = UnifiedModulesStrategy;
});
//# sourceMappingURL=data:application/json;base64,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