/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getClosureSafeProperty } from '../../util/property';
/**
 * Construct an `InjectableDef` which defines how a token will be constructed by the DI system, and
 * in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call `inject` to access the `Injector` and request injection of dependencies.
 *
 * @codeGenApi
 */
export function ɵɵdefineInjectable(opts) {
    return {
        token: opts.token,
        providedIn: opts.providedIn || null,
        factory: opts.factory,
        value: undefined,
    };
}
/**
 * @deprecated in v8, delete after v10. This API should be used only be generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export var defineInjectable = ɵɵdefineInjectable;
/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `factory`: an `InjectorType` is an instantiable type, so a zero argument `factory` function to
 *   create the type must be provided. If that factory function needs to inject arguments, it can
 *   use the `inject` function.
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @publicApi
 */
export function ɵɵdefineInjector(options) {
    return {
        factory: options.factory,
        providers: options.providers || [],
        imports: options.imports || [],
    };
}
/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export function getInjectableDef(type) {
    return getOwnDefinition(type, type[NG_PROV_DEF]) ||
        getOwnDefinition(type, type[NG_INJECTABLE_DEF]);
}
/**
 * Return `def` only if it is defined directly on `type` and is not inherited from a base
 * class of `type`.
 *
 * The function `Object.hasOwnProperty` is not sufficient to distinguish this case because in older
 * browsers (e.g. IE10) static property inheritance is implemented by copying the properties.
 *
 * Instead, the definition's `token` is compared to the `type`, and if they don't match then the
 * property was not defined directly on the type itself, and was likely inherited. The definition
 * is only returned if the `type` matches the `def.token`.
 */
function getOwnDefinition(type, def) {
    return def && def.token === type ? def : null;
}
/**
 * Read the injectable def (`ɵprov`) for `type` or read the `ɵprov` from one of its ancestors.
 *
 * @param type A type which may have `ɵprov`, via inheritance.
 *
 * @deprecated Will be removed in v10, where an error will occur in the scenario if we find the
 * `ɵprov` on an ancestor only.
 */
export function getInheritedInjectableDef(type) {
    // See `jit/injectable.ts#compileInjectable` for context on NG_PROV_DEF_FALLBACK.
    var def = type &&
        (type[NG_PROV_DEF] || type[NG_INJECTABLE_DEF] ||
            (type[NG_PROV_DEF_FALLBACK] && type[NG_PROV_DEF_FALLBACK]()));
    if (def) {
        var typeName = getTypeName(type);
        // TODO(FW-1307): Re-add ngDevMode when closure can handle it
        // ngDevMode &&
        console.warn("DEPRECATED: DI is instantiating a token \"" + typeName + "\" that inherits its @Injectable decorator but does not provide one itself.\n" +
            ("This will become an error in v10. Please add @Injectable() to the \"" + typeName + "\" class."));
        return def;
    }
    else {
        return null;
    }
}
/** Gets the name of a type, accounting for some cross-browser differences. */
function getTypeName(type) {
    // `Function.prototype.name` behaves differently between IE and other browsers. In most browsers
    // it'll always return the name of the function itself, no matter how many other functions it
    // inherits from. On IE the function doesn't have its own `name` property, but it takes it from
    // the lowest level in the prototype chain. E.g. if we have `class Foo extends Parent` most
    // browsers will evaluate `Foo.name` to `Foo` while IE will return `Parent`. We work around
    // the issue by converting the function to a string and parsing its name out that way via a regex.
    if (type.hasOwnProperty('name')) {
        return type.name;
    }
    var match = ('' + type).match(/^function\s*([^\s(]+)/);
    return match === null ? '' : match[1];
}
/**
 * Read the injector def type in a way which is immune to accidentally reading inherited value.
 *
 * @param type type which may have an injector def (`ɵinj`)
 */
export function getInjectorDef(type) {
    return type && (type.hasOwnProperty(NG_INJ_DEF) || type.hasOwnProperty(NG_INJECTOR_DEF)) ?
        type[NG_INJ_DEF] :
        null;
}
export var NG_PROV_DEF = getClosureSafeProperty({ ɵprov: getClosureSafeProperty });
export var NG_INJ_DEF = getClosureSafeProperty({ ɵinj: getClosureSafeProperty });
// On IE10 properties defined via `defineProperty` won't be inherited by child classes,
// which will break inheriting the injectable definition from a grandparent through an
// undecorated parent class. We work around it by defining a fallback method which will be
// used to retrieve the definition. This should only be a problem in JIT mode, because in
// AOT TypeScript seems to have a workaround for static properties. When inheriting from an
// undecorated parent is no longer supported in v10, this can safely be removed.
export var NG_PROV_DEF_FALLBACK = getClosureSafeProperty({ ɵprovFallback: getClosureSafeProperty });
// We need to keep these around so we can read off old defs if new defs are unavailable
export var NG_INJECTABLE_DEF = getClosureSafeProperty({ ngInjectableDef: getClosureSafeProperty });
export var NG_INJECTOR_DEF = getClosureSafeProperty({ ngInjectorDef: getClosureSafeProperty });
//# sourceMappingURL=data:application/json;base64,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