/*
 * Copyright 2019-2020 by Security and Safety Things GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.securityandsafetythings.webserver;

import android.content.Context;
import android.support.annotation.NonNull;
import android.util.Log;
import android.util.Pair;
import android.webkit.MimeTypeMap;
import com.google.common.collect.ImmutableMap;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.securityandsafetythings.webserver.utilities.InstantSerializer;
import com.securityandsafetythings.webserver.utilities.RestMethodWrapper;
import com.securityandsafetythings.webserver.utilities.RestPath;
import com.securityandsafetythings.webserver.utilities.SharedMemoryFactory;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;

import java.io.File;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import static com.securityandsafetythings.webserver.WebServerResponse.ResponseStatus.REDIRECT_SEE_OTHER;

/**
 * Implements a web request handler that you can feed with JAX-RS style annotated methods.
 * You can create an instance of it, use the {@link #register(Object)} method to add JAX-RS style handlers,
 * (for example RestEndPoint) and then register this instance as a web server using
 * the Web Server Manager (or WebServerConnector in this example).
 * See MainService for an example.
 */
public class RestHandler implements RequestHandler {
    private static final String LOGTAG = RestHandler.class.getSimpleName();
    private static final Gson GSON = new GsonBuilder().registerTypeAdapter(Instant.class, new InstantSerializer())
        .create();
    private static final String REST_PATH = "/rest";
    private static final String SLASH = "/";
    private static final String TEXT_PLAIN = "text/plain";
    @SuppressWarnings("MagicNumber")
    private static final int HTTP_REDIRECT_CODE_MIN = 300;
    @SuppressWarnings("MagicNumber")
    private static final int HTTP_REDIRECT_CODE_MAX = 308;
    private static final String EMPTY_PATH = "";
    private static final String JAVASCRIPT_EXT = "js";

    private final RestPath mGetRoutes = new RestPath("GET");
    private final RestPath mPutRoutes = new RestPath("PUT");
    private final RestPath mPostRoutes = new RestPath("POST");
    private final RestPath mDeleteRoutes = new RestPath("DELETE");
    private final Map<String, Pair<String, WebServerResponse.ResponseStatus>> mRedirections = new HashMap<>();
    private final Set<Object> mRestServices = new HashSet<>();
    private final SharedMemoryFactory mSharedMemoryFactory = new SharedMemoryFactory();
    private final Context mContext;
    private final String mBasePath;
    private final String mWebsiteAssetPath;
    private boolean mServeIndexOnAnyRoute = false;

    /**
     * Constructor accepts {@link Context} parameter and uses it to create the basePath
     *
     * @param c context
     * @param websiteAssetPath path to store website assets = website
     */
    public RestHandler(final Context c, final String websiteAssetPath) {
        mContext = c;
        mBasePath = File.separator + "app" + File.separator + c.getPackageName();
        mWebsiteAssetPath = websiteAssetPath;
    }

    /**
     * Converts from json to T
     *
     * @param json input Json format string.
     * @param type the target type to convert to.
     * @param <T>  the expected type
     * @return the converted object
     */
    public static <T> T fromJson(final String json, final Type type) {
        return GSON.fromJson(json, type);
    }

    /**
     * Convert from an object to Json format string.
     *
     * @param object the object to convert.
     * @return Json format string.
     */
    public static String toJson(final Object object) {
        return GSON.toJson(object);
    }

    /**
     * Register a new object to handle requests. See RestEndPoint for an example of
     * a suitable object.
     *
     * @param restService The instance to register. Must be filled with methods annotated
     *                    with JAX-RS annotations.
     * @param <T>         The type of the service
     */
    @SuppressWarnings("unchecked")
    public <T> void register(@NonNull final T restService) {
        register(restService, (Class<T>)restService.getClass());
    }

    private <T> void register(final T restService, final Class<? super T> clazz) {
        if (mRestServices.contains(restService)) {
            return;
        }
        mRestServices.add(restService);
        final Path classPath = clazz.getAnnotation(Path.class);
        for (final Method method : clazz.getMethods()) {
            final Path methodPath = method.getAnnotation(Path.class);
            final String route = REST_PATH + pathToRoute(classPath) + pathToRoute(methodPath);
            if (method.isAnnotationPresent(GET.class)) {
                Log.v(LOGTAG, String.format("Registering %s#%s for route GET %s", clazz.getSimpleName(), method.getName(), route));
                mGetRoutes.addMethod(route, new RestMethodWrapper(
                    restService, method, route, mContext.getCacheDir(), mSharedMemoryFactory));
            } else if (method.isAnnotationPresent(PUT.class)) {
                Log.v(LOGTAG, String.format("Registering %s#%s for route PUT %s", clazz.getSimpleName(), method.getName(), route));
                mPutRoutes.addMethod(route, new RestMethodWrapper(
                    restService, method, route, mContext.getCacheDir(), mSharedMemoryFactory));
            } else if (method.isAnnotationPresent(POST.class)) {
                Log.v(LOGTAG, String.format("Registering %s#%s for route POST %s", clazz.getSimpleName(), method.getName(), route));
                mPostRoutes.addMethod(route, new RestMethodWrapper(
                    restService, method, route, mContext.getCacheDir(), mSharedMemoryFactory));
            } else if (method.isAnnotationPresent(DELETE.class)) {
                Log.v(LOGTAG, String.format("Registering %s#%s for route DELETE %s", clazz.getSimpleName(), method.getName(), route));
                mDeleteRoutes.addMethod(route, new RestMethodWrapper(restService, method, route, mContext.getCacheDir(),
                    mSharedMemoryFactory));
            }
        }
    }

    /**
     * This will cause static content serving to return /index.html for any path.
     * <p>
     * This is useful for single-page applications that do routing by themselves.
     */
    public void serveIndexOnAnyRoute() {
        mServeIndexOnAnyRoute = true;
    }

    /**
     * Register a redirection (303 SEE OTHER) from a specific path to another one.
     * See {@link #registerRedirect(String, String, WebServerResponse.ResponseStatus)}
     *
     * @param from the path to redirect.
     * @param to   the path that should be redirected to.
     */
    public void registerRedirect(final String from, final String to) {
        registerRedirect(from, to, REDIRECT_SEE_OTHER);
    }

    /**
     * Register a redirection from a specific path to another one.
     *
     * @param from          The path to redirect from. Is always a local path, can be prefixed with "/" or not.
     * @param to            The redirection target. Can be an absolute path ("https://www.google.com"),
     *                      a device path ("/app/com.sast.test/something"), or a local path ("rest/example/info").
     * @param redirectType: The type of redirect to perform (300 - 308).
     */
    private void registerRedirect(final String from, final String to, final WebServerResponse.ResponseStatus redirectType) {
        if (redirectType.getRequestStatus() < HTTP_REDIRECT_CODE_MIN || redirectType.getRequestStatus() > HTTP_REDIRECT_CODE_MAX) {
            throw new IllegalArgumentException("Invalid redirect response: " + redirectType);
        }
        if (from.startsWith(SLASH)) {
            mRedirections.put(from, new Pair<>(to, redirectType));
        } else {
            mRedirections.put(SLASH + from, new Pair<>(to, redirectType));
        }
    }

    @Override
    public WebServerResponse onGet(final WebServerRequest webServerRequest) {
        final String route = getPath(webServerRequest);
        final Pair<String, WebServerResponse.ResponseStatus> redirectionPair = mRedirections.get(route);
        if (redirectionPair != null) {
            return createRedirectResponse(redirectionPair.first, redirectionPair.second);
        }

        if (route.startsWith(REST_PATH)) {
            return invokeMethod(mGetRoutes, route, webServerRequest);
        } else {
            if ("".equals(route)) {
                //Timber.d("Redirecting empty path to /");
                return createRedirectResponse(mBasePath + SLASH, REDIRECT_SEE_OTHER);
            }
            String resource = route.substring(1);
            final String mimeType = resolveMimeType(resource);
            final WebServerResponse response = WebServerResponse.createAssetResponse(mContext,
                mWebsiteAssetPath + SLASH + resource,
                WebServerResponse.ResponseStatus.OK, mimeType, Collections.emptyMap());
            if (response != null) {
                return response;
            } else if (SLASH.equals(route) || (mServeIndexOnAnyRoute && mimeType.startsWith("text/"))) {
                // if we don't find the asset, and its media type is text (which is default if no other type matches),
                // load index.html innoCompress stead
                Log.d(LOGTAG, String.format("Trying to load index.html for unknown %s", resource));
                resource = "index.html";
                return WebServerResponse.createAssetResponse(mContext, mWebsiteAssetPath + SLASH + resource,
                    WebServerResponse.ResponseStatus.OK, resolveMimeType(resource), Collections.emptyMap());
            }
            return null;
        }
    }

    @Override
    public WebServerResponse onPut(final WebServerRequest webServerRequest) {
        final String route = getPath(webServerRequest);
        return invokeMethod(mPutRoutes, route, webServerRequest);
    }

    @Override
    public WebServerResponse onPost(final WebServerRequest webServerRequest) {
        final String route = getPath(webServerRequest);
        return invokeMethod(mPostRoutes, route, webServerRequest);
    }

    @Override
    public WebServerResponse onDelete(final WebServerRequest webServerRequest) {
        final String route = getPath(webServerRequest);
        return invokeMethod(mDeleteRoutes, route, webServerRequest);
    }

    private WebServerResponse invokeMethod(final RestPath restPath, final String route, final WebServerRequest webServerRequest) {
        Log.d(LOGTAG, String.format("%s %s", restPath.getPathNode(), route));
        final String normalizedRoute = route.startsWith(SLASH) ? route.substring(1) : route;
        return restPath.resolve(normalizedRoute)
            .map(restMethodWrapper1 -> restMethodWrapper1.invoke(webServerRequest, normalizedRoute))
            .orElse(null);
    }

    private String getPath(final WebServerRequest webServerRequest) {
        String path = SLASH;
        if (webServerRequest.getUri() != null && webServerRequest.getUri()
            .getPath() != null) {
            path = webServerRequest.getUri()
                .getPath()
                .replace(mBasePath, EMPTY_PATH);
        }
        return path;
    }

    private String pathToRoute(final Path path) {
        if (path != null) {
            return path.value()
                .startsWith(SLASH) ? path.value() : SLASH + path.value();
        }
        return EMPTY_PATH;
    }

    private String resolveMimeType(@NonNull final String route) {
        final String fileExtension = MimeTypeMap.getFileExtensionFromUrl(route);
        String mimeType = MimeTypeMap.getSingleton()
            .getMimeTypeFromExtension(fileExtension);
        if (mimeType == null && JAVASCRIPT_EXT.equals(fileExtension)) {
            mimeType = "application/javascript";
        }
        if (mimeType == null) {
            mimeType = TEXT_PLAIN;
        }
        return mimeType;
    }

    private WebServerResponse createRedirectResponse(final String to, final WebServerResponse.ResponseStatus type) {
        return WebServerResponse.createStringResponse(
            EMPTY_PATH,
            type,
            TEXT_PLAIN,
            ImmutableMap.of("location", to)
        );
    }
}
