package com.securityandsafetythings.examples.aiapp.aicore.aiprocess;

import android.app.Application;
import android.content.Context;
import android.util.Log;

import androidx.annotation.NonNull;

import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiprocess.aicallback.AiProcessCallback;
import com.securityandsafetythings.examples.aiapp.aicore.mediapool.MediaNode;

import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.Nullable;

import java.util.concurrent.LinkedBlockingDeque;

public abstract class AIProcess extends Thread{
    static String LOGTAG = AIProcess.class.getSimpleName();
    protected Context mContext;
    protected Application mApplication;

    /** Queue process pipeline */
    private final LinkedBlockingDeque<InferenceResult> inputQueue;
    private static final int MAX_ITEM_IN_PROCESS_DEQUEUE = 2;

    /** Flag for thread safe */
    private Boolean isProcess = true;
    private AIProcess nextProcess;

    /** Thread statement */
    protected float fps = 0;
    protected long lastTimeProcess = System.currentTimeMillis();

    /** Callback for ai statement & output utilities */
    protected AiProcessCallback aiProcessCallback = null;

    public AIProcess(Context mContext, Application mApplication, AIProcess nextProcess) {
        this.nextProcess = nextProcess;
        this.mContext = mContext;
        this.mApplication = mApplication;
        this.inputQueue = new LinkedBlockingDeque<>();

        aiProcessCallback = new AiProcessCallback() {
            @Override
            public void onAiProcessDetectNothing(InferenceResult inferenceResult) {}
            @Override
            public void onAiProcessDetectSomething(InferenceResult inferenceResult, InferenceResult outputInference) {}
            @Override
            public void onAiProcessTerminate() {}
        };
    }


    /** This constructor use for last ai-process in pipeline */
   /* public AIProcess(Context mContext, Application mApplication) {
        this.mContext = mContext;
        this.mApplication = mApplication;
        this.inputQueue = new LinkedBlockingDeque<>();
    }*/

    /** setup aiProcess callback -> maybe useful for motion trigger */
    public void setAIProcessCallback(AiProcessCallback aiProcessCallback){
        Log.d(LOGTAG, "AiProcessCallback check instance = " + aiProcessCallback);
        this.aiProcessCallback = aiProcessCallback;
    }

    @Override
    public void run() {
        while (isProcess){
            if (!inputQueue.isEmpty()){
                /* calculate fps runtime */
                long currentTimeProcess = System.currentTimeMillis();
                fps = 1000f/(float) (currentTimeProcess - lastTimeProcess);
                lastTimeProcess = currentTimeProcess;

                /* get element from queue then process */
                try {
                    InferenceResult inputInference = inputQueue.takeFirst();
                    //Log.d("MediaPool_", "take first bmp =" + inputInference.getProcessBitmap().hashCode() + " on getting from queue ai");
                    InferenceResult outputInference = onProcess(inputInference);
                    if (outputInference != null){
                        if (nextProcess != null ){
                            //Log.d("MediaPool_", "media node forward to next thread");
                            nextProcess.addItem(outputInference);
                        } else {
                            //Callback on detect something
                            aiProcessCallback.onAiProcessDetectSomething(inputInference, outputInference);
                            //Log.d("MediaPool_", "free bmp =" + inputInference.getProcessBitmap().hashCode() + " on finish pipeline");
                            /* Free the inputReference if inputReference forward to the end off pipeline */
                            inputInference.freeMediaNode();
                        }
                    } else {
                        //Callback on nothing detected at runtime
                        aiProcessCallback.onAiProcessDetectNothing(inputInference);
                        /* Free the inputReference if the outputInference null -> current inputReference not forward to any ai process and not use anymore */
                        //Log.d("MediaPool_", "free bmp =" + inputInference.getProcessBitmap().hashCode() + " on null output reference");
                        inputInference.freeMediaNode();
                    }
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            } else {
                /* implement wait - notify mechanism */
                synchronized (this) {
                    try {
                        this.wait();
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
            }
        }

        /* on End Process STEP */
        onProcessEnd();
        //release all queue's elements and free their mediaNode
        clearInputQueueProcess();
        Log.d(LOGTAG + "_checkReleaseProcess", "thread " + this.getName() +" terminated");
        aiProcessCallback.onAiProcessTerminate();
    }

    public boolean addItem(InferenceResult inputInference) throws InterruptedException {
        if (inputQueue.size() >= MAX_ITEM_IN_PROCESS_DEQUEUE) {
            InferenceResult removeInference = inputQueue.takeFirst();
            /* Free the inputReference if queue reach max size, ai-process skip inputInference */
            //Log.d("MediaPool_", "free bmp =" + removeInference.getProcessBitmap().hashCode() + " on queue ai reach max size");
            removeInference.freeMediaNode();

        }
        boolean isAdded = inputQueue.offerLast(inputInference);

        /* implement wait() - notify() mechanism */
        synchronized (this) {
            this.notify();
        }
        return isAdded;
    }

    /** This method called if this instance is the first pipeline algorithm */
    public boolean addItem(@NonNull MediaNode inputMedia) throws InterruptedException {
        InferenceResult inputInference = new InferenceResult(inputMedia) {
            @Nullable
            @Contract(pure = true)
            @Override
            public ResultName getResultName() {
                return null;
            }
        };
        return addItem(inputInference);
    }

    /** force end process and wait until thread completely terminated
     * Output: return this for other thread can call join() for sync */
    public AIProcess endProcess(){
        Log.d(LOGTAG + "_checkReleaseProcess", "start end process: " + LOGTAG);
        //synchronized (isProcess){
        //Log.d(LOGTAG + "_checkReleaseProcess", "set isProcess flag to false");

        synchronized (this) {
            isProcess = false;
            this.notify(); // in case thread is in waiting-state
        }
        Log.d(LOGTAG + "_checkReleaseProcess", "release process finish");
        return this;
    }

    /** Only call on Process end: release all queue's element & free their Media Node*/
    private void clearInputQueueProcess(){
        while (!inputQueue.isEmpty()){
            try {
                InferenceResult releaseInference = inputQueue.takeFirst();
                releaseInference.freeMediaNode();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    public abstract InferenceResult onProcess(InferenceResult inputInference);
    public abstract void onProcessEnd();
}
