package com.securityandsafetythings.examples.aiapp.aicore.mediapool;

import android.graphics.Bitmap;

import androidx.annotation.NonNull;

import org.bytedeco.javacv.Frame;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class MediaPool {
    String LOGTAG = MediaPool.class.getSimpleName();
    /** Variables for manage pool */
    private MediaNode headMediaNode = null;
    private MediaNode tailMediaNode = null;
    public int minSize = 10;
    private Integer size = 0;

    /** Utilities for frame converter */
    ByteBuffer buffer;
    byte[] row;

    public MediaPool() {}

    public MediaPool(String deviceId, int minSize) {
        this.minSize = minSize;
        LOGTAG += "_" + deviceId;
    }


    public synchronized void addFirst(MediaNode mediaNode){
        if (size == 0){
            /** if list mediaNode empty -> first element have backward & forward null */
            headMediaNode = mediaNode;
            tailMediaNode = mediaNode;
            size++;
        } else {
            /** list mediaNode not empty -> last element still null, set current head element forward
             *  and new head element backward */
            mediaNode.backwardPtr = this.headMediaNode;
            this.headMediaNode.forwardPtr = mediaNode;
            this.headMediaNode = mediaNode;
            size++;
        }
    }

    /*public void addFirst(MediaNode mediaNode){
        synchronized (size) {
            synchronized (headMediaNode) {
                if (size == 0) {
                    headMediaNode = mediaNode;
                    tailMediaNode = mediaNode;
                    size++;
                } else {
                    mediaNode.backwardPtr = this.headMediaNode;
                    this.headMediaNode.forwardPtr = mediaNode;
                    this.headMediaNode = mediaNode;
                    size++;
                }
            }
        }
    }*/

    /** Pop mediaNode on tail list if free else return null */
    public MediaNode takeFreeMediaNode(){
        synchronized (size) {
            /* return null MediaNode when current list mediaNode not reaches min size */
            if (size < minSize) {
                return null;
            }
        }

        synchronized (tailMediaNode) {
            /* return null mediaNode when tail mediaNode still been referenced */
            //Log.d("MediaPool_", "check free mediaNode, tail = " + tailMediaNode.getBitmap().hashCode() + " num refer = " + tailMediaNode.getReference()+ "~" + tailMediaNode.isReference());
            if (tailMediaNode.isReference()) {
                //Log.d("checkMediaPool_takeFree", "tail still been referred");
                return null;
            } else {
                MediaNode currentTail = tailMediaNode;
                tailMediaNode = tailMediaNode.forwardPtr;
                size--;
                //Log.d("checkMediaPool_takeFree", "got free element");

                //clear all freeNode's reference ptr
                currentTail.forwardPtr = null;
                currentTail.backwardPtr = null;
                return currentTail;
            }
        }
    }

    /** Count the number of current free mediaNode from tail */
    public int getNumFreeMediaNode(){
        MediaNode mediaNode = null;
        synchronized (tailMediaNode) {
            mediaNode = tailMediaNode;
        }
        int count = 0;

        while (!mediaNode.isReference()){
            mediaNode = mediaNode.forwardPtr;
            count++;
        }
        return count;
    }

    /** provide newest frame for encode live-view, ai-process, recording
     * -> when an object call getRealtimeMediaNode(), this will return First mediaNode and also increase reference
     * of that mediaNode
     * */
    public MediaNode getRealtimeMediaNode(){
        synchronized (size) {
            if (size == 0) {
                return null;
            }
        }
        synchronized (headMediaNode) {
            headMediaNode.takeReference();
            return headMediaNode;
        }
    }

    public int size(){
        synchronized (size) {
            return size;
        }
    }

    @Override
    public String toString() {
        String s = "";
        MediaNode mediaNode;
        mediaNode = headMediaNode;
        while (mediaNode != null){
            s += "[" /*+  mediaNode.getFrameId()*/ + ":" + mediaNode.getReference() + "]";
            mediaNode = mediaNode.backwardPtr;
        }
        return s;
    }

    /** >>>>>>>>>>>>>>>>>>>>>>> Utilities method for frame convert <<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */
    ByteBuffer gray2rgba(ByteBuffer in, int width, int height, int stride, int rowBytes) {
        if (buffer == null || buffer.capacity() < height * rowBytes) {
            buffer = ByteBuffer.allocate(height * rowBytes);
        }
        if (row == null || row.length != stride)
            row = new byte[stride];
        for (int y = 0; y < height; y++) {
            in.position(y * stride);
            in.get(row);
            for (int x = 0; x < width; x++) {
                // GRAY -> RGBA
                byte B = row[x];
                int rgba = (B & 0xff) << 24 |
                        (B & 0xff) << 16 |
                        (B & 0xff) <<  8 | 0xff;
                buffer.putInt(y * rowBytes + 4 * x, rgba);
            }
        }
        return buffer;
    }

    ByteBuffer bgr2rgba(@NonNull ByteBuffer in, int width, int height, int stride, int rowBytes) {
        if (!in.order().equals(ByteOrder.LITTLE_ENDIAN)) {
            in = in.order(ByteOrder.LITTLE_ENDIAN);
        }
        if (buffer == null || buffer.capacity() < height * rowBytes) {
            buffer = ByteBuffer.allocate(height * rowBytes);
        }
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++) {
                // BGR -> RGBA
                int rgb;
                if (x < width - 1 || y < height - 1) {
                    rgb = in.getInt(y * stride + 3 * x);
                } else {
                    int b = in.get(y * stride + 3 * x    ) & 0xff;
                    int g = in.get(y * stride + 3 * x + 1) & 0xff;
                    int r = in.get(y * stride + 3 * x + 2) & 0xff;
                    rgb = (r << 16) | (g << 8) | b;
                }
                buffer.putInt(y * rowBytes + 4 * x, (rgb << 8) | 0xff);
            }
        }
        return buffer;
    }

    public void convert(Frame frame) {
        if (frame == null || frame.image == null) {
            return ;
        }

        Bitmap.Config config = null;
        switch (frame.imageChannels) {
            case 2: config = Bitmap.Config.RGB_565; break;
            case 1:
            case 3:
            case 4: config = Bitmap.Config.ARGB_8888; break;
            default: assert false;
        }

        MediaNode mediaNode = this.takeFreeMediaNode();
        Bitmap bitmap = null;
        if (mediaNode == null){
            bitmap = Bitmap.createBitmap(frame.imageWidth, frame.imageHeight, config);
            mediaNode = new MediaNode(bitmap);
        } else {
            bitmap = mediaNode.getBitmap();
            mediaNode.setFrameId(System.currentTimeMillis());
        }

        // assume frame.imageDepth == Frame.DEPTH_UBYTE
        ByteBuffer in = (ByteBuffer)frame.image[0];
        int width = frame.imageWidth;
        int height = frame.imageHeight;
        int stride = frame.imageStride;
        int rowBytes = bitmap.getRowBytes();
        if (frame.imageChannels == 1) {
            gray2rgba(in, width, height, stride, rowBytes);
            bitmap.copyPixelsFromBuffer(buffer.position(0));
        } else if (frame.imageChannels == 3) {
            bgr2rgba(in, width, height, stride, rowBytes);
            bitmap.copyPixelsFromBuffer(buffer.position(0));
        } else {
            // assume matching strides
            bitmap.copyPixelsFromBuffer(in.position(0));
        }

        //Log.d("checkMediaPool_converter", "mediaNode=" + mediaNode.toString() /*+ " bitmap=" + mediaNode.getBitmap().toString()*/ + " frameId=" +mediaNode.getFrameId() );
        //Log.d("checkMediaPool_converter", "current state = " + this.toString() );

        this.addFirst(mediaNode);
        //Log.w("newFrameDecoded", "Head node id = " + mediaNode.getFrameId() );
        //Log.d("checkMediaPool_converter", "finish decode frame in" );
        //return mediaNode;
    }
}
