/*
 * Copyright 2019-2020 by Security and Safety Things GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.securityandsafetythings.examples.aiapp;

import android.graphics.Bitmap;
import android.media.Image;
import com.securityandsafetythings.examples.aiapp.utilities.BitmapUtilities;
import com.securityandsafetythings.webserver.utilities.ProducesHeader;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;


/**
 * Class responsible to receive API calls from the front end, process it, and return the result.
 */
@Path("example")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class RestEndPoint {
    private byte[] mBitmapBytes;
    private byte[] rtspBitMapBytes;
    private byte[] rtsp2BitMapBytes;
    private byte[] rtsp3BitMapBytes;
    /**
     * Default empty constructor to obtain a class instance.
     */
    public RestEndPoint() {

    }

    /**
     * Simple endpoint used to test the connection on the front end.
     *
     * @return String declaring a connection was made
     */
    @GET
    @Path("test")
    @Produces(MediaType.APPLICATION_JSON)
    public String testConnection() {
        return "Test Successful";
    }

    /**
     * Sets the most recently received {@link Image} from the Video pipeline.
     *
     * @param bitmap the {@link Image} retrieved from the video pipeline
     */
    public synchronized void setImage(final Bitmap bitmap) {
        mBitmapBytes = BitmapUtilities.compressBitmap(bitmap);
    }
    public synchronized void setRTSPFrame(final Bitmap bitmap) { rtspBitMapBytes = BitmapUtilities.compressBitmap(bitmap); }
    public synchronized void setRTSP2Frame(final Bitmap bitmap) { rtsp2BitMapBytes = BitmapUtilities.compressBitmap(bitmap); }
    public synchronized void setRTSP3Frame(final Bitmap bitmap) { rtsp3BitMapBytes = BitmapUtilities.compressBitmap(bitmap); }

    /**
     * Returns the most recent video pipeline {@link Image} as a byte[].
     *
     * @return byte[] the converted byte[] of the {@link Bitmap}
     */
    @SuppressWarnings("MagicNumber")
    @GET
    @Path("live")
    @Produces("image/jpeg")
    @ProducesHeader("Cache-Control: max-age=5")
    public synchronized byte[] getImage() {
        if (mBitmapBytes == null) {
            //throw new NotFoundException();
        }
        return mBitmapBytes;
    }
    /**
     * Returns RTSP stream from another IP Camera as a byte[]
     */
    @SuppressWarnings("MagicNumber")
    @GET
    @Path("rtsp")
    @Produces("image/jpeg")
    @ProducesHeader("Cache-Control: max-age=5")
    public synchronized byte[] getRTSP1Frame() {
        if (rtspBitMapBytes == null) {
            //throw new NotFoundException();
        }
        return rtspBitMapBytes;
    }
    /**
     * Returns RTSP stream from another IP Camera as a byte[]
     */
    @SuppressWarnings("MagicNumber")
    @GET
    @Path("rtsp2")
    @Produces("image/jpeg")
    @ProducesHeader("Cache-Control: max-age=5")
    public synchronized byte[] getRTSP2Frame() {
        if (rtsp2BitMapBytes == null) {
            //throw new NotFoundException();
        }
        return rtsp2BitMapBytes;
    }
    /**
     * Returns RTSP stream from another IP Camera as a byte[]
     */
    @SuppressWarnings("MagicNumber")
    @GET
    @Path("rtsp3")
    @Produces("image/jpeg")
    @ProducesHeader("Cache-Control: max-age=5")
    public synchronized byte[] getRTSP3Frame() {
        if (rtsp3BitMapBytes == null) {
            //throw new NotFoundException();
        }
        return rtsp3BitMapBytes;
    }
}
