package com.securityandsafetythings.examples.aiapp.aicore.aiLibs.algorithm.nativewarpper.motion;

import android.graphics.Bitmap;
import android.graphics.Matrix;

import androidx.annotation.NonNull;

import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.InferenceResult;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.aiInference.detector.Bbox;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.algorithm.nativewarpper.NativeInference;
import com.securityandsafetythings.examples.aiapp.aicore.aiLibs.algorithm.nativewarpper.sorttrack.TrackBox;

import org.jetbrains.annotations.NotNull;
import org.opencv.android.Utils;
import org.opencv.core.Core;
import org.opencv.core.Mat;
import org.opencv.core.Point;
import org.opencv.core.Rect;
import org.opencv.core.Size;
import org.opencv.imgproc.Imgproc;

public class MotionDetector extends NativeInference {
    static final String LOGTAG = MotionDetector.class.getSimpleName();

    /** native function */
    private native void initNative(String algoName, String key);
    private native boolean nativeOnImageMotion(long imageMat, long fgMat, long bgMat, int thresholdMotionPixel, String key, boolean nightMode);
    private native void releaseNativeMotionDetector(String deviceId);
    private String key;

    /** motion algorithm attribute use for utilities */
    private static float scaleImage = 0.22f;
    //private static final int        MOTION_IMAGE_WIDTH = 432;
    //private static final int        MOTION_IMAGE_HEIGHT = 243;
    //private static final double[]   BG_COLOR = new double[]{0.0};
    private static final Size       KERNAL_SIZE = new Size(4,4);

    /** Constance */
    private static final String     ALGO_NAME = "ViBe";
    private static final float      MIN_THRES_NIGHTMODE = 0.02f;
    private static final float      ACCEPTABLE_THRESHOLD_NIGHTMODE = 2f;//0.1f;
    private static final float      MIN_THRESH_DEFAULT = 0.1f;
    private static final float      ACCEPTABLE_THRESHOLD_DEFAULT = 0.3f;
    private final int               MAX_FRAME_CHANGEMODE = 10;

    /** attribute for flag control */
    private Mode modeRuntime = Mode.RGB_MODE;
    private int countFrameChangeMode = 0;
    private static long DELAY_MOTION_ACTIVE = 5000;
    protected long lastTimeProcess = System.currentTimeMillis();

    /** Define runtime motionDetector mode */
    enum Mode {
        GRAYSCALE_MODE("gray"),
        RGB_MODE("rgb"); // rgb or bgr

        private String modeType;
        Mode(String type) {
            this.modeType = type;
        }
    }

    public MotionDetector(boolean nightMode){
        this.modeRuntime = nightMode ? Mode.GRAYSCALE_MODE : Mode.RGB_MODE;
        key = String.valueOf(System.currentTimeMillis());
        initNative(ALGO_NAME, key);
    }

    @Override
    public MotionResult runInference(@NotNull InferenceResult inputInference) {
        Bitmap imageBmp = inputInference.getProcessBitmap();

        /** motion scale 1: scale by image ratio */
        float scaleWidth = scaleImage;
        float scaleHeight = scaleImage;

        /** motion scale 2: scale by image fix size */
        //float scaleWidth = (float)MOTION_IMAGE_WIDTH / (float)imageBmp.getWidth();
        //float scaleHeight = (float)MOTION_IMAGE_HEIGHT / (float)imageBmp.getHeight();

        // RESIZE THE BIT MAP
        final Matrix matrix = new Matrix();
        matrix.postScale(scaleWidth, scaleHeight);
        final Bitmap inputBitmap = Bitmap.createBitmap(imageBmp,
                0,
                0,
                imageBmp.getWidth(),
                imageBmp.getHeight(),
                matrix,
                false);

        // pre-pair Mat input/output
        Mat imageMat = new Mat();
        Utils.bitmapToMat(inputBitmap, imageMat);
        Mat rgbImageMat = new Mat();
        Imgproc.cvtColor(imageMat, rgbImageMat, Imgproc.COLOR_BGRA2BGR);

        // noise image filter
        Imgproc.blur(rgbImageMat,rgbImageMat, KERNAL_SIZE);

        Mat bgMat = new Mat();
        Mat fgMat = new Mat();
        int areaThreshold = (int)(inputBitmap.getHeight() * 0.037 * inputBitmap.getWidth() * 0.037);

        boolean nightMode = (this.modeRuntime == Mode.GRAYSCALE_MODE);
        boolean result = nativeOnImageMotion(rgbImageMat.getNativeObjAddr(), fgMat.getNativeObjAddr(), bgMat.getNativeObjAddr(), areaThreshold, key, nightMode);

        if (checkIsGrayScaleImage(rgbImageMat)){
            resetByMode(Mode.GRAYSCALE_MODE);
        } else {
            resetByMode(Mode.RGB_MODE);
        }

        return new MotionResult(inputInference, fgMat, result, nightMode);
    }

    @Override
    public void release() {
        releaseNativeMotionDetector(key);
    }

    private boolean checkIsGrayScaleImage(Mat imageRGB){
        /** detect if input image is grayScale by check 5-pixel in inputMat */
        boolean checkCenter = isGrayPixel(imageRGB, imageRGB.cols() / 2, imageRGB.rows() /2);
        boolean checkCornrer1 = isGrayPixel(imageRGB, imageRGB.cols() / 8, imageRGB.rows() / 8);
        boolean checkCornrer2 = isGrayPixel(imageRGB, imageRGB.cols() * 7 / 8, imageRGB.rows() * 7 / 8);
        boolean checkCornrer3 = isGrayPixel(imageRGB, imageRGB.cols() / 8, imageRGB.rows() * 7 / 8);
        boolean checkCornrer4 = isGrayPixel(imageRGB, imageRGB.cols() * 7 / 8, imageRGB.rows() / 8);

        return checkCenter && checkCornrer1 && checkCornrer2 && checkCornrer3 && checkCornrer4;
    }

    public boolean resetByMode(Mode runtimeMode){
        if (this.modeRuntime != runtimeMode){
            // check reach max frame to change mode
            if (countFrameChangeMode > MAX_FRAME_CHANGEMODE) {
                // change mode confirm -> reset count
                countFrameChangeMode = 0;
                release();
                //logger.i(LOGTAG, "reset motion detector in " + deviceId + " for Mode: " + runtimeMode.name());
                this.modeRuntime = runtimeMode;
                initNative(ALGO_NAME, this.key);

                //Log.d("_checkMotion_modeReset", "reset vibe");
                return true;
            } else{
                countFrameChangeMode++;
                //Log.d("_checkMotion_modeReset", "wait for reaching max frame gray to reset vibe");
            }
        } else {
            // cancel change mode -> reset count
            //Log.d("_checkMotion_modeReset", "cancel reset vibe with a frame still " + runtimeMode.name());
            countFrameChangeMode = 0;
        }
        return false;
    }

    private boolean isGrayPixel(Mat imageRGB, int col, int row){
        double[] colors = imageRGB.get(row,col);
        if (colors.length == 3){
            return colors[0] == colors[1] && colors[1] == colors[2];
        } else return colors.length == 1;
    }

    public boolean getRuntimeMode() {
        return (this.modeRuntime == Mode.GRAYSCALE_MODE);
    }

    /** Utilities method */
    public static float getMotionScoreOfTrack(@NonNull Mat fgImage, @NonNull TrackBox trackBox){
        Bbox bbox = (Bbox) trackBox.getInferenceResult(InferenceResult.ResultName.box);
        //Log.d("_checkBoxMotion", "real box size:" + (int)bbox.mLocation.left + " " + (int)bbox.mLocation.right + " " + (int)bbox.mLocation.top + " "  + (int)bbox.mLocation.bottom );
        //Log.d(LOGTAG, "confirmed bbox = (" + bbox.x1 + "," + bbox.y1 +") (" + bbox.x2 + "," + bbox.y2 + ")" );
        int x1 = (int) (bbox.x1 * scaleImage);
        int x2 = (int) (bbox.x2 * scaleImage);
        int y1 = (int) (bbox.y1 * scaleImage);
        int y2 = (int) (bbox.y2 * scaleImage);

        /** Scenario 1: Calculate motion score by java core -> deprecated because bad runtime cpu: 1 -> >100ms with box size big */
        //long startCalaulateByJavaCore = System.currentTimeMillis();
        /*int totalWhitePoint = 0;
        //Log.d("_checkBoxMotion", "motion = " + Arrays.toString(fgImage.get(x1,y1)));
        for (int i = x1; i <= x2; ++i){
            for (int j = y1 ;j <= y2; ++j){
                if (fgImage.get(j,i) != null && fgImage.get(j,i)[0] != 0){
                    totalWhitePoint++;
                }
            }
        }
        float result = (float)totalWhitePoint / (float)((x2 - x1) * (y2 - y1));*/
        //long runtimeByJavaCore = System.currentTimeMillis() - startCalaulateByJavaCore;

        /** Scenario 2: calculate motion score by opencv count nonZero */
        //long startCalculateByCV = System.currentTimeMillis();
        Rect rectCrop = new Rect(new Point(x1,y1), new Point(x2,y2));
        Mat croppedMat = fgImage.submat(rectCrop);

        int totalWhitePoint = Core.countNonZero(croppedMat);
        //long runtimeByCV = System.currentTimeMillis() - startCalculateByCV;

        //Log.d(LOGTAG, "compare 2 scenario calculate motionScore = " +totalWhitePoint + " ~?~ " + total + " | runtime = " + runtimeByJavaCore + " ~/~ " + runtimeByCV);
        //Log.d("_checkBoxMotion",  deviceId + " box size:" + (x2 - x1) + " " + (y2 - y1) + " " + ((x2 - x1) * (y2 - y1) * motionThreshold) + " motion value = " + totalWhitePoint);
        return (float)totalWhitePoint / (float)((x2 - x1) * (y2 - y1));
    }

    @NonNull
    public static boolean[] checkStateMotionImage(float motionScore, boolean nightMode){
        boolean passMin;
        boolean passAcceptable;
        if (nightMode){
            passMin = motionScore > MIN_THRES_NIGHTMODE;
            passAcceptable = motionScore > ACCEPTABLE_THRESHOLD_NIGHTMODE;
            //Log.d("_checkMotion", "check nightMode with Face motion score = " + motionScore);
        } else {
            passMin = motionScore > MIN_THRESH_DEFAULT;
            passAcceptable = motionScore > ACCEPTABLE_THRESHOLD_DEFAULT;
        }
        return new boolean[]{passMin, passAcceptable};
    }

}
