/**
 * @fileoverview added by tsickle
 * Generated from: packages/common/http/src/client.ts
 * @suppress {checkTypes,constantProperty,extraRequire,missingOverride,missingReturn,unusedPrivateMembers,uselessCode} checked by tsc
 */
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Injectable } from '@angular/core';
import { of } from 'rxjs';
import { concatMap, filter, map } from 'rxjs/operators';
import { HttpHandler } from './backend';
import { HttpHeaders } from './headers';
import { HttpParams } from './params';
import { HttpRequest } from './request';
import { HttpResponse } from './response';
/**
 * Constructs an instance of `HttpRequestOptions<T>` from a source `HttpMethodOptions` and
 * the given `body`. This function clones the object and adds the body.
 *
 * Note that the `responseType` *options* value is a String that identifies the
 * single data type of the response.
 * A single overload version of the method handles each response type.
 * The value of `responseType` cannot be a union, as the combined signature could imply.
 *
 * @template T
 * @param {?} options
 * @param {?} body
 * @return {?}
 */
function addBody(options, body) {
    return {
        body,
        headers: options.headers,
        observe: options.observe,
        params: options.params,
        reportProgress: options.reportProgress,
        responseType: options.responseType,
        withCredentials: options.withCredentials,
    };
}
/**
 * Performs HTTP requests.
 * This service is available as an injectable class, with methods to perform HTTP requests.
 * Each request method has multiple signatures, and the return type varies based on
 * the signature that is called (mainly the values of `observe` and `responseType`).
 *
 * Note that the `responseType` *options* value is a String that identifies the
 * single data type of the response.
 * A single overload version of the method handles each response type.
 * The value of `responseType` cannot be a union, as the combined signature could imply.
 *
 * \@usageNotes
 * Sample HTTP requests for the [Tour of Heroes](/tutorial/toh-pt0) application.
 *
 * ### HTTP Request Example
 *
 * ```
 *  // GET heroes whose name contains search term
 * searchHeroes(term: string): observable<Hero[]>{
 *
 *  const params = new HttpParams({fromString: 'name=term'});
 *    return this.httpClient.request('GET', this.heroesUrl, {responseType:'json', params});
 * }
 * ```
 * ### JSONP Example
 * ```
 * requestJsonp(url, callback = 'callback') {
 *  return this.httpClient.jsonp(this.heroesURL, callback);
 * }
 * ```
 *
 * ### PATCH Example
 * ```
 * // PATCH one of the heroes' name
 * patchHero (id: number, heroName: string): Observable<{}> {
 * const url = `${this.heroesUrl}/${id}`;   // PATCH api/heroes/42
 *  return this.httpClient.patch(url, {name: heroName}, httpOptions)
 *    .pipe(catchError(this.handleError('patchHero')));
 * }
 * ```
 *
 * @see [HTTP Guide](guide/http)
 *
 * \@publicApi
 */
export class HttpClient {
    /**
     * @param {?} handler
     */
    constructor(handler) {
        this.handler = handler;
    }
    /**
     * Constructs an observable for a generic HTTP request that, when subscribed,
     * fires the request through the chain of registered interceptors and on to the
     * server.
     *
     * You can pass an `HttpRequest` directly as the only parameter. In this case,
     * the call returns an observable of the raw `HttpEvent` stream.
     *
     * Alternatively you can pass an HTTP method as the first parameter,
     * a URL string as the second, and an options hash containing the request body as the third.
     * See `addBody()`. In this case, the specified `responseType` and `observe` options determine the
     * type of returned observable.
     *   * The `responseType` value determines how a successful response body is parsed.
     *   * If `responseType` is the default `json`, you can pass a type interface for the resulting
     * object as a type parameter to the call.
     *
     * The `observe` value determines the return type, according to what you are interested in
     * observing.
     *   * An `observe` value of events returns an observable of the raw `HttpEvent` stream, including
     * progress events by default.
     *   * An `observe` value of response returns an observable of `HttpResponse<T>`,
     * where the `T` parameter depends on the `responseType` and any optionally provided type
     * parameter.
     *   * An `observe` value of body returns an observable of `<T>` with the same `T` body type.
     *
     * @param {?} first
     * @param {?=} url
     * @param {?=} options
     * @return {?}
     */
    request(first, url, options = {}) {
        /** @type {?} */
        let req;
        // First, check whether the primary argument is an instance of `HttpRequest`.
        if (first instanceof HttpRequest) {
            // It is. The other arguments must be undefined (per the signatures) and can be
            // ignored.
            req = first;
        }
        else {
            // It's a string, so it represents a URL. Construct a request based on it,
            // and incorporate the remaining arguments (assuming `GET` unless a method is
            // provided.
            // Figure out the headers.
            /** @type {?} */
            let headers = undefined;
            if (options.headers instanceof HttpHeaders) {
                headers = options.headers;
            }
            else {
                headers = new HttpHeaders(options.headers);
            }
            // Sort out parameters.
            /** @type {?} */
            let params = undefined;
            if (!!options.params) {
                if (options.params instanceof HttpParams) {
                    params = options.params;
                }
                else {
                    params = new HttpParams((/** @type {?} */ ({ fromObject: options.params })));
                }
            }
            // Construct the request.
            req = new HttpRequest(first, (/** @type {?} */ (url)), (options.body !== undefined ? options.body : null), {
                headers,
                params,
                reportProgress: options.reportProgress,
                // By default, JSON is assumed to be returned for all calls.
                responseType: options.responseType || 'json',
                withCredentials: options.withCredentials,
            });
        }
        // Start with an Observable.of() the initial request, and run the handler (which
        // includes all interceptors) inside a concatMap(). This way, the handler runs
        // inside an Observable chain, which causes interceptors to be re-run on every
        // subscription (this also makes retries re-run the handler, including interceptors).
        /** @type {?} */
        const events$ = of(req).pipe(concatMap((/**
         * @param {?} req
         * @return {?}
         */
        (req) => this.handler.handle(req))));
        // If coming via the API signature which accepts a previously constructed HttpRequest,
        // the only option is to get the event stream. Otherwise, return the event stream if
        // that is what was requested.
        if (first instanceof HttpRequest || options.observe === 'events') {
            return events$;
        }
        // The requested stream contains either the full response or the body. In either
        // case, the first step is to filter the event stream to extract a stream of
        // responses(s).
        /** @type {?} */
        const res$ = (/** @type {?} */ (events$.pipe(filter((/**
         * @param {?} event
         * @return {?}
         */
        (event) => event instanceof HttpResponse)))));
        // Decide which stream to return.
        switch (options.observe || 'body') {
            case 'body':
                // The requested stream is the body. Map the response stream to the response
                // body. This could be done more simply, but a misbehaving interceptor might
                // transform the response body into a different format and ignore the requested
                // responseType. Guard against this by validating that the response is of the
                // requested type.
                switch (req.responseType) {
                    case 'arraybuffer':
                        return res$.pipe(map((/**
                         * @param {?} res
                         * @return {?}
                         */
                        (res) => {
                            // Validate that the body is an ArrayBuffer.
                            if (res.body !== null && !(res.body instanceof ArrayBuffer)) {
                                throw new Error('Response is not an ArrayBuffer.');
                            }
                            return res.body;
                        })));
                    case 'blob':
                        return res$.pipe(map((/**
                         * @param {?} res
                         * @return {?}
                         */
                        (res) => {
                            // Validate that the body is a Blob.
                            if (res.body !== null && !(res.body instanceof Blob)) {
                                throw new Error('Response is not a Blob.');
                            }
                            return res.body;
                        })));
                    case 'text':
                        return res$.pipe(map((/**
                         * @param {?} res
                         * @return {?}
                         */
                        (res) => {
                            // Validate that the body is a string.
                            if (res.body !== null && typeof res.body !== 'string') {
                                throw new Error('Response is not a string.');
                            }
                            return res.body;
                        })));
                    case 'json':
                    default:
                        // No validation needed for JSON responses, as they can be of any type.
                        return res$.pipe(map((/**
                         * @param {?} res
                         * @return {?}
                         */
                        (res) => res.body)));
                }
            case 'response':
                // The response stream was requested directly, so return it.
                return res$;
            default:
                // Guard against new future observe types being added.
                throw new Error(`Unreachable: unhandled observe type ${options.observe}}`);
        }
    }
    /**
     * Constructs an observable that, when subscribed, causes the configured
     * `DELETE` request to execute on the server. See the individual overloads for
     * details on the return type.
     *
     * @param {?} url     The endpoint URL.
     * @param {?=} options The HTTP options to send with the request.
     *
     * @return {?}
     */
    delete(url, options = {}) {
        return this.request('DELETE', url, (/** @type {?} */ (options)));
    }
    /**
     * Constructs an observable that, when subscribed, causes the configured
     * `GET` request to execute on the server. See the individual overloads for
     * details on the return type.
     * @param {?} url
     * @param {?=} options
     * @return {?}
     */
    get(url, options = {}) {
        return this.request('GET', url, (/** @type {?} */ (options)));
    }
    /**
     * Constructs an observable that, when subscribed, causes the configured
     * `HEAD` request to execute on the server. The `HEAD` method returns
     * meta information about the resource without transferring the
     * resource itself. See the individual overloads for
     * details on the return type.
     * @param {?} url
     * @param {?=} options
     * @return {?}
     */
    head(url, options = {}) {
        return this.request('HEAD', url, (/** @type {?} */ (options)));
    }
    /**
     * Constructs an `Observable` that, when subscribed, causes a request with the special method
     * `JSONP` to be dispatched via the interceptor pipeline.
     * The [JSONP pattern](https://en.wikipedia.org/wiki/JSONP) works around limitations of certain
     * API endpoints that don't support newer,
     * and preferable [CORS](https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS) protocol.
     * JSONP treats the endpoint API as a JavaScript file and tricks the browser to process the
     * requests even if the API endpoint is not located on the same domain (origin) as the client-side
     * application making the request.
     * The endpoint API must support JSONP callback for JSONP requests to work.
     * The resource API returns the JSON response wrapped in a callback function.
     * You can pass the callback function name as one of the query parameters.
     * Note that JSONP requests can only be used with `GET` requests.
     *
     * @template T
     * @param {?} url The resource URL.
     * @param {?} callbackParam The callback function name.
     *
     * @return {?}
     */
    jsonp(url, callbackParam) {
        return this.request('JSONP', url, {
            params: new HttpParams().append(callbackParam, 'JSONP_CALLBACK'),
            observe: 'body',
            responseType: 'json',
        });
    }
    /**
     * Constructs an `Observable` that, when subscribed, causes the configured
     * `OPTIONS` request to execute on the server. This method allows the client
     * to determine the supported HTTP methods and other capabilites of an endpoint,
     * without implying a resource action. See the individual overloads for
     * details on the return type.
     * @param {?} url
     * @param {?=} options
     * @return {?}
     */
    options(url, options = {}) {
        return this.request('OPTIONS', url, (/** @type {?} */ (options)));
    }
    /**
     * Constructs an observable that, when subscribed, causes the configured
     * `PATCH` request to execute on the server. See the individual overloads for
     * details on the return type.
     * @param {?} url
     * @param {?} body
     * @param {?=} options
     * @return {?}
     */
    patch(url, body, options = {}) {
        return this.request('PATCH', url, addBody(options, body));
    }
    /**
     * Constructs an observable that, when subscribed, causes the configured
     * `POST` request to execute on the server. The server responds with the location of
     * the replaced resource. See the individual overloads for
     * details on the return type.
     * @param {?} url
     * @param {?} body
     * @param {?=} options
     * @return {?}
     */
    post(url, body, options = {}) {
        return this.request('POST', url, addBody(options, body));
    }
    /**
     * Constructs an observable that, when subscribed, causes the configured
     * `PUT` request to execute on the server. The `PUT` method replaces an existing resource
     * with a new set of values.
     * See the individual overloads for details on the return type.
     * @param {?} url
     * @param {?} body
     * @param {?=} options
     * @return {?}
     */
    put(url, body, options = {}) {
        return this.request('PUT', url, addBody(options, body));
    }
}
HttpClient.decorators = [
    { type: Injectable }
];
/** @nocollapse */
HttpClient.ctorParameters = () => [
    { type: HttpHandler }
];
if (false) {
    /**
     * @type {?}
     * @private
     */
    HttpClient.prototype.handler;
}
//# sourceMappingURL=data:application/json;base64,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