/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { createTokenForExternalReference, Identifiers } from '../identifiers';
import { LifecycleHooks } from '../lifecycle_reflector';
import * as o from '../output/output_ast';
import { convertValueToOutputAst } from '../output/value_util';
import { ProviderAstType } from '../template_parser/template_ast';
export function providerDef(ctx, providerAst) {
    let flags = 0 /* None */;
    if (!providerAst.eager) {
        flags |= 4096 /* LazyProvider */;
    }
    if (providerAst.providerType === ProviderAstType.PrivateService) {
        flags |= 8192 /* PrivateProvider */;
    }
    if (providerAst.isModule) {
        flags |= 1073741824 /* TypeModuleProvider */;
    }
    providerAst.lifecycleHooks.forEach((lifecycleHook) => {
        // for regular providers, we only support ngOnDestroy
        if (lifecycleHook === LifecycleHooks.OnDestroy ||
            providerAst.providerType === ProviderAstType.Directive ||
            providerAst.providerType === ProviderAstType.Component) {
            flags |= lifecycleHookToNodeFlag(lifecycleHook);
        }
    });
    const { providerExpr, flags: providerFlags, depsExpr } = providerAst.multiProvider ?
        multiProviderDef(ctx, flags, providerAst.providers) :
        singleProviderDef(ctx, flags, providerAst.providerType, providerAst.providers[0]);
    return {
        providerExpr,
        flags: providerFlags,
        depsExpr,
        tokenExpr: tokenExpr(ctx, providerAst.token),
    };
}
function multiProviderDef(ctx, flags, providers) {
    const allDepDefs = [];
    const allParams = [];
    const exprs = providers.map((provider, providerIndex) => {
        let expr;
        if (provider.useClass) {
            const depExprs = convertDeps(providerIndex, provider.deps || provider.useClass.diDeps);
            expr = ctx.importExpr(provider.useClass.reference).instantiate(depExprs);
        }
        else if (provider.useFactory) {
            const depExprs = convertDeps(providerIndex, provider.deps || provider.useFactory.diDeps);
            expr = ctx.importExpr(provider.useFactory.reference).callFn(depExprs);
        }
        else if (provider.useExisting) {
            const depExprs = convertDeps(providerIndex, [{ token: provider.useExisting }]);
            expr = depExprs[0];
        }
        else {
            expr = convertValueToOutputAst(ctx, provider.useValue);
        }
        return expr;
    });
    const providerExpr = o.fn(allParams, [new o.ReturnStatement(o.literalArr(exprs))], o.INFERRED_TYPE);
    return {
        providerExpr,
        flags: flags | 1024 /* TypeFactoryProvider */,
        depsExpr: o.literalArr(allDepDefs)
    };
    function convertDeps(providerIndex, deps) {
        return deps.map((dep, depIndex) => {
            const paramName = `p${providerIndex}_${depIndex}`;
            allParams.push(new o.FnParam(paramName, o.DYNAMIC_TYPE));
            allDepDefs.push(depDef(ctx, dep));
            return o.variable(paramName);
        });
    }
}
function singleProviderDef(ctx, flags, providerType, providerMeta) {
    let providerExpr;
    let deps;
    if (providerType === ProviderAstType.Directive || providerType === ProviderAstType.Component) {
        providerExpr = ctx.importExpr(providerMeta.useClass.reference);
        flags |= 16384 /* TypeDirective */;
        deps = providerMeta.deps || providerMeta.useClass.diDeps;
    }
    else {
        if (providerMeta.useClass) {
            providerExpr = ctx.importExpr(providerMeta.useClass.reference);
            flags |= 512 /* TypeClassProvider */;
            deps = providerMeta.deps || providerMeta.useClass.diDeps;
        }
        else if (providerMeta.useFactory) {
            providerExpr = ctx.importExpr(providerMeta.useFactory.reference);
            flags |= 1024 /* TypeFactoryProvider */;
            deps = providerMeta.deps || providerMeta.useFactory.diDeps;
        }
        else if (providerMeta.useExisting) {
            providerExpr = o.NULL_EXPR;
            flags |= 2048 /* TypeUseExistingProvider */;
            deps = [{ token: providerMeta.useExisting }];
        }
        else {
            providerExpr = convertValueToOutputAst(ctx, providerMeta.useValue);
            flags |= 256 /* TypeValueProvider */;
            deps = [];
        }
    }
    const depsExpr = o.literalArr(deps.map(dep => depDef(ctx, dep)));
    return { providerExpr, flags, depsExpr };
}
function tokenExpr(ctx, tokenMeta) {
    return tokenMeta.identifier ? ctx.importExpr(tokenMeta.identifier.reference) :
        o.literal(tokenMeta.value);
}
export function depDef(ctx, dep) {
    // Note: the following fields have already been normalized out by provider_analyzer:
    // - isAttribute, isHost
    const expr = dep.isValue ? convertValueToOutputAst(ctx, dep.value) : tokenExpr(ctx, dep.token);
    let flags = 0 /* None */;
    if (dep.isSkipSelf) {
        flags |= 1 /* SkipSelf */;
    }
    if (dep.isOptional) {
        flags |= 2 /* Optional */;
    }
    if (dep.isSelf) {
        flags |= 4 /* Self */;
    }
    if (dep.isValue) {
        flags |= 8 /* Value */;
    }
    return flags === 0 /* None */ ? expr : o.literalArr([o.literal(flags), expr]);
}
export function lifecycleHookToNodeFlag(lifecycleHook) {
    let nodeFlag = 0 /* None */;
    switch (lifecycleHook) {
        case LifecycleHooks.AfterContentChecked:
            nodeFlag = 2097152 /* AfterContentChecked */;
            break;
        case LifecycleHooks.AfterContentInit:
            nodeFlag = 1048576 /* AfterContentInit */;
            break;
        case LifecycleHooks.AfterViewChecked:
            nodeFlag = 8388608 /* AfterViewChecked */;
            break;
        case LifecycleHooks.AfterViewInit:
            nodeFlag = 4194304 /* AfterViewInit */;
            break;
        case LifecycleHooks.DoCheck:
            nodeFlag = 262144 /* DoCheck */;
            break;
        case LifecycleHooks.OnChanges:
            nodeFlag = 524288 /* OnChanges */;
            break;
        case LifecycleHooks.OnDestroy:
            nodeFlag = 131072 /* OnDestroy */;
            break;
        case LifecycleHooks.OnInit:
            nodeFlag = 65536 /* OnInit */;
            break;
    }
    return nodeFlag;
}
export function componentFactoryResolverProviderDef(reflector, ctx, flags, entryComponents) {
    const entryComponentFactories = entryComponents.map((entryComponent) => ctx.importExpr(entryComponent.componentFactory));
    const token = createTokenForExternalReference(reflector, Identifiers.ComponentFactoryResolver);
    const classMeta = {
        diDeps: [
            { isValue: true, value: o.literalArr(entryComponentFactories) },
            { token: token, isSkipSelf: true, isOptional: true },
            { token: createTokenForExternalReference(reflector, Identifiers.NgModuleRef) },
        ],
        lifecycleHooks: [],
        reference: reflector.resolveExternalReference(Identifiers.CodegenComponentFactoryResolver)
    };
    const { providerExpr, flags: providerFlags, depsExpr } = singleProviderDef(ctx, flags, ProviderAstType.PrivateService, {
        token,
        multi: false,
        useClass: classMeta,
    });
    return { providerExpr, flags: providerFlags, depsExpr, tokenExpr: tokenExpr(ctx, token) };
}
//# sourceMappingURL=data:application/json;base64,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