/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { visitValue } from '../util';
import * as o from './output_ast';
export const QUOTED_KEYS = '$quoted$';
export function convertValueToOutputAst(ctx, value, type = null) {
    return visitValue(value, new _ValueOutputAstTransformer(ctx), type);
}
class _ValueOutputAstTransformer {
    constructor(ctx) {
        this.ctx = ctx;
    }
    visitArray(arr, type) {
        const values = [];
        // Note Array.map() must not be used to convert the values because it will
        // skip over empty elements in arrays constructed using `new Array(length)`,
        // resulting in `undefined` elements. This breaks the type guarantee that
        // all values in `o.LiteralArrayExpr` are of type `o.Expression`.
        // See test case in `value_util_spec.ts`.
        for (let i = 0; i < arr.length; ++i) {
            values.push(visitValue(arr[i], this, null /* context */));
        }
        return o.literalArr(values, type);
    }
    visitStringMap(map, type) {
        const entries = [];
        const quotedSet = new Set(map && map[QUOTED_KEYS]);
        Object.keys(map).forEach(key => {
            entries.push(new o.LiteralMapEntry(key, visitValue(map[key], this, null), quotedSet.has(key)));
        });
        return new o.LiteralMapExpr(entries, type);
    }
    visitPrimitive(value, type) {
        return o.literal(value, type);
    }
    visitOther(value, type) {
        if (value instanceof o.Expression) {
            return value;
        }
        else {
            return this.ctx.importExpr(value);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmFsdWVfdXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvbXBpbGVyL3NyYy9vdXRwdXQvdmFsdWVfdXRpbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFHSCxPQUFPLEVBQWtDLFVBQVUsRUFBQyxNQUFNLFNBQVMsQ0FBQztBQUVwRSxPQUFPLEtBQUssQ0FBQyxNQUFNLGNBQWMsQ0FBQztBQUVsQyxNQUFNLENBQUMsTUFBTSxXQUFXLEdBQUcsVUFBVSxDQUFDO0FBRXRDLE1BQU0sVUFBVSx1QkFBdUIsQ0FDbkMsR0FBa0IsRUFBRSxLQUFVLEVBQUUsT0FBb0IsSUFBSTtJQUMxRCxPQUFPLFVBQVUsQ0FBQyxLQUFLLEVBQUUsSUFBSSwwQkFBMEIsQ0FBQyxHQUFHLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUN0RSxDQUFDO0FBRUQsTUFBTSwwQkFBMEI7SUFDOUIsWUFBb0IsR0FBa0I7UUFBbEIsUUFBRyxHQUFILEdBQUcsQ0FBZTtJQUFHLENBQUM7SUFDMUMsVUFBVSxDQUFDLEdBQVUsRUFBRSxJQUFZO1FBQ2pDLE1BQU0sTUFBTSxHQUFtQixFQUFFLENBQUM7UUFDbEMsMEVBQTBFO1FBQzFFLDRFQUE0RTtRQUM1RSx5RUFBeUU7UUFDekUsaUVBQWlFO1FBQ2pFLHlDQUF5QztRQUN6QyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsRUFBRTtZQUNuQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO1NBQzNEO1FBQ0QsT0FBTyxDQUFDLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQsY0FBYyxDQUFDLEdBQXlCLEVBQUUsSUFBZTtRQUN2RCxNQUFNLE9BQU8sR0FBd0IsRUFBRSxDQUFDO1FBQ3hDLE1BQU0sU0FBUyxHQUFHLElBQUksR0FBRyxDQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUMzRCxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUM3QixPQUFPLENBQUMsSUFBSSxDQUNSLElBQUksQ0FBQyxDQUFDLGVBQWUsQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLEVBQUUsU0FBUyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDeEYsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQyxDQUFDLGNBQWMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVELGNBQWMsQ0FBQyxLQUFVLEVBQUUsSUFBWTtRQUNyQyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRCxVQUFVLENBQUMsS0FBVSxFQUFFLElBQVk7UUFDakMsSUFBSSxLQUFLLFlBQVksQ0FBQyxDQUFDLFVBQVUsRUFBRTtZQUNqQyxPQUFPLEtBQUssQ0FBQztTQUNkO2FBQU07WUFDTCxPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ25DO0lBQ0gsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBJbmMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5cbmltcG9ydCB7T3V0cHV0Q29udGV4dCwgVmFsdWVUcmFuc2Zvcm1lciwgdmlzaXRWYWx1ZX0gZnJvbSAnLi4vdXRpbCc7XG5cbmltcG9ydCAqIGFzIG8gZnJvbSAnLi9vdXRwdXRfYXN0JztcblxuZXhwb3J0IGNvbnN0IFFVT1RFRF9LRVlTID0gJyRxdW90ZWQkJztcblxuZXhwb3J0IGZ1bmN0aW9uIGNvbnZlcnRWYWx1ZVRvT3V0cHV0QXN0KFxuICAgIGN0eDogT3V0cHV0Q29udGV4dCwgdmFsdWU6IGFueSwgdHlwZTogby5UeXBlfG51bGwgPSBudWxsKTogby5FeHByZXNzaW9uIHtcbiAgcmV0dXJuIHZpc2l0VmFsdWUodmFsdWUsIG5ldyBfVmFsdWVPdXRwdXRBc3RUcmFuc2Zvcm1lcihjdHgpLCB0eXBlKTtcbn1cblxuY2xhc3MgX1ZhbHVlT3V0cHV0QXN0VHJhbnNmb3JtZXIgaW1wbGVtZW50cyBWYWx1ZVRyYW5zZm9ybWVyIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSBjdHg6IE91dHB1dENvbnRleHQpIHt9XG4gIHZpc2l0QXJyYXkoYXJyOiBhbnlbXSwgdHlwZTogby5UeXBlKTogby5FeHByZXNzaW9uIHtcbiAgICBjb25zdCB2YWx1ZXM6IG8uRXhwcmVzc2lvbltdID0gW107XG4gICAgLy8gTm90ZSBBcnJheS5tYXAoKSBtdXN0IG5vdCBiZSB1c2VkIHRvIGNvbnZlcnQgdGhlIHZhbHVlcyBiZWNhdXNlIGl0IHdpbGxcbiAgICAvLyBza2lwIG92ZXIgZW1wdHkgZWxlbWVudHMgaW4gYXJyYXlzIGNvbnN0cnVjdGVkIHVzaW5nIGBuZXcgQXJyYXkobGVuZ3RoKWAsXG4gICAgLy8gcmVzdWx0aW5nIGluIGB1bmRlZmluZWRgIGVsZW1lbnRzLiBUaGlzIGJyZWFrcyB0aGUgdHlwZSBndWFyYW50ZWUgdGhhdFxuICAgIC8vIGFsbCB2YWx1ZXMgaW4gYG8uTGl0ZXJhbEFycmF5RXhwcmAgYXJlIG9mIHR5cGUgYG8uRXhwcmVzc2lvbmAuXG4gICAgLy8gU2VlIHRlc3QgY2FzZSBpbiBgdmFsdWVfdXRpbF9zcGVjLnRzYC5cbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IGFyci5sZW5ndGg7ICsraSkge1xuICAgICAgdmFsdWVzLnB1c2godmlzaXRWYWx1ZShhcnJbaV0sIHRoaXMsIG51bGwgLyogY29udGV4dCAqLykpO1xuICAgIH1cbiAgICByZXR1cm4gby5saXRlcmFsQXJyKHZhbHVlcywgdHlwZSk7XG4gIH1cblxuICB2aXNpdFN0cmluZ01hcChtYXA6IHtba2V5OiBzdHJpbmddOiBhbnl9LCB0eXBlOiBvLk1hcFR5cGUpOiBvLkV4cHJlc3Npb24ge1xuICAgIGNvbnN0IGVudHJpZXM6IG8uTGl0ZXJhbE1hcEVudHJ5W10gPSBbXTtcbiAgICBjb25zdCBxdW90ZWRTZXQgPSBuZXcgU2V0PHN0cmluZz4obWFwICYmIG1hcFtRVU9URURfS0VZU10pO1xuICAgIE9iamVjdC5rZXlzKG1hcCkuZm9yRWFjaChrZXkgPT4ge1xuICAgICAgZW50cmllcy5wdXNoKFxuICAgICAgICAgIG5ldyBvLkxpdGVyYWxNYXBFbnRyeShrZXksIHZpc2l0VmFsdWUobWFwW2tleV0sIHRoaXMsIG51bGwpLCBxdW90ZWRTZXQuaGFzKGtleSkpKTtcbiAgICB9KTtcbiAgICByZXR1cm4gbmV3IG8uTGl0ZXJhbE1hcEV4cHIoZW50cmllcywgdHlwZSk7XG4gIH1cblxuICB2aXNpdFByaW1pdGl2ZSh2YWx1ZTogYW55LCB0eXBlOiBvLlR5cGUpOiBvLkV4cHJlc3Npb24ge1xuICAgIHJldHVybiBvLmxpdGVyYWwodmFsdWUsIHR5cGUpO1xuICB9XG5cbiAgdmlzaXRPdGhlcih2YWx1ZTogYW55LCB0eXBlOiBvLlR5cGUpOiBvLkV4cHJlc3Npb24ge1xuICAgIGlmICh2YWx1ZSBpbnN0YW5jZW9mIG8uRXhwcmVzc2lvbikge1xuICAgICAgcmV0dXJuIHZhbHVlO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gdGhpcy5jdHguaW1wb3J0RXhwcih2YWx1ZSk7XG4gICAgfVxuICB9XG59XG4iXX0=