/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ElementRef } from '../linker/element_ref';
import { QueryList } from '../linker/query_list';
import { asElementData, asProviderData, asQueryList } from './types';
import { declaredViewContainer, filterQueryId, isEmbeddedView } from './util';
export function queryDef(flags, id, bindings) {
    var bindingDefs = [];
    for (var propName in bindings) {
        var bindingType = bindings[propName];
        bindingDefs.push({ propName: propName, bindingType: bindingType });
    }
    return {
        // will bet set by the view definition
        nodeIndex: -1,
        parent: null,
        renderParent: null,
        bindingIndex: -1,
        outputIndex: -1,
        // regular values
        // TODO(vicb): check
        checkIndex: -1,
        flags: flags,
        childFlags: 0,
        directChildFlags: 0,
        childMatchedQueries: 0,
        ngContentIndex: -1,
        matchedQueries: {},
        matchedQueryIds: 0,
        references: {},
        childCount: 0,
        bindings: [],
        bindingFlags: 0,
        outputs: [],
        element: null,
        provider: null,
        text: null,
        query: { id: id, filterId: filterQueryId(id), bindings: bindingDefs },
        ngContent: null
    };
}
export function createQuery() {
    return new QueryList();
}
export function dirtyParentQueries(view) {
    var queryIds = view.def.nodeMatchedQueries;
    while (view.parent && isEmbeddedView(view)) {
        var tplDef = view.parentNodeDef;
        view = view.parent;
        // content queries
        var end = tplDef.nodeIndex + tplDef.childCount;
        for (var i = 0; i <= end; i++) {
            var nodeDef = view.def.nodes[i];
            if ((nodeDef.flags & 67108864 /* TypeContentQuery */) &&
                (nodeDef.flags & 536870912 /* DynamicQuery */) &&
                (nodeDef.query.filterId & queryIds) === nodeDef.query.filterId) {
                asQueryList(view, i).setDirty();
            }
            if ((nodeDef.flags & 1 /* TypeElement */ && i + nodeDef.childCount < tplDef.nodeIndex) ||
                !(nodeDef.childFlags & 67108864 /* TypeContentQuery */) ||
                !(nodeDef.childFlags & 536870912 /* DynamicQuery */)) {
                // skip elements that don't contain the template element or no query.
                i += nodeDef.childCount;
            }
        }
    }
    // view queries
    if (view.def.nodeFlags & 134217728 /* TypeViewQuery */) {
        for (var i = 0; i < view.def.nodes.length; i++) {
            var nodeDef = view.def.nodes[i];
            if ((nodeDef.flags & 134217728 /* TypeViewQuery */) && (nodeDef.flags & 536870912 /* DynamicQuery */)) {
                asQueryList(view, i).setDirty();
            }
            // only visit the root nodes
            i += nodeDef.childCount;
        }
    }
}
export function checkAndUpdateQuery(view, nodeDef) {
    var queryList = asQueryList(view, nodeDef.nodeIndex);
    if (!queryList.dirty) {
        return;
    }
    var directiveInstance;
    var newValues = undefined;
    if (nodeDef.flags & 67108864 /* TypeContentQuery */) {
        var elementDef = nodeDef.parent.parent;
        newValues = calcQueryValues(view, elementDef.nodeIndex, elementDef.nodeIndex + elementDef.childCount, nodeDef.query, []);
        directiveInstance = asProviderData(view, nodeDef.parent.nodeIndex).instance;
    }
    else if (nodeDef.flags & 134217728 /* TypeViewQuery */) {
        newValues = calcQueryValues(view, 0, view.def.nodes.length - 1, nodeDef.query, []);
        directiveInstance = view.component;
    }
    queryList.reset(newValues);
    var bindings = nodeDef.query.bindings;
    var notify = false;
    for (var i = 0; i < bindings.length; i++) {
        var binding = bindings[i];
        var boundValue = void 0;
        switch (binding.bindingType) {
            case 0 /* First */:
                boundValue = queryList.first;
                break;
            case 1 /* All */:
                boundValue = queryList;
                notify = true;
                break;
        }
        directiveInstance[binding.propName] = boundValue;
    }
    if (notify) {
        queryList.notifyOnChanges();
    }
}
function calcQueryValues(view, startIndex, endIndex, queryDef, values) {
    for (var i = startIndex; i <= endIndex; i++) {
        var nodeDef = view.def.nodes[i];
        var valueType = nodeDef.matchedQueries[queryDef.id];
        if (valueType != null) {
            values.push(getQueryValue(view, nodeDef, valueType));
        }
        if (nodeDef.flags & 1 /* TypeElement */ && nodeDef.element.template &&
            (nodeDef.element.template.nodeMatchedQueries & queryDef.filterId) ===
                queryDef.filterId) {
            var elementData = asElementData(view, i);
            // check embedded views that were attached at the place of their template,
            // but process child nodes first if some match the query (see issue #16568)
            if ((nodeDef.childMatchedQueries & queryDef.filterId) === queryDef.filterId) {
                calcQueryValues(view, i + 1, i + nodeDef.childCount, queryDef, values);
                i += nodeDef.childCount;
            }
            if (nodeDef.flags & 16777216 /* EmbeddedViews */) {
                var embeddedViews = elementData.viewContainer._embeddedViews;
                for (var k = 0; k < embeddedViews.length; k++) {
                    var embeddedView = embeddedViews[k];
                    var dvc = declaredViewContainer(embeddedView);
                    if (dvc && dvc === elementData) {
                        calcQueryValues(embeddedView, 0, embeddedView.def.nodes.length - 1, queryDef, values);
                    }
                }
            }
            var projectedViews = elementData.template._projectedViews;
            if (projectedViews) {
                for (var k = 0; k < projectedViews.length; k++) {
                    var projectedView = projectedViews[k];
                    calcQueryValues(projectedView, 0, projectedView.def.nodes.length - 1, queryDef, values);
                }
            }
        }
        if ((nodeDef.childMatchedQueries & queryDef.filterId) !== queryDef.filterId) {
            // if no child matches the query, skip the children.
            i += nodeDef.childCount;
        }
    }
    return values;
}
export function getQueryValue(view, nodeDef, queryValueType) {
    if (queryValueType != null) {
        // a match
        switch (queryValueType) {
            case 1 /* RenderElement */:
                return asElementData(view, nodeDef.nodeIndex).renderElement;
            case 0 /* ElementRef */:
                return new ElementRef(asElementData(view, nodeDef.nodeIndex).renderElement);
            case 2 /* TemplateRef */:
                return asElementData(view, nodeDef.nodeIndex).template;
            case 3 /* ViewContainerRef */:
                return asElementData(view, nodeDef.nodeIndex).viewContainer;
            case 4 /* Provider */:
                return asProviderData(view, nodeDef.nodeIndex).instance;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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