/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { EMPTY_ARRAY, EMPTY_OBJ } from './empty';
import { NG_COMP_DEF, NG_DIR_DEF, NG_FACTORY_DEF, NG_LOC_ID_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
var _renderCompCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(function () {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        var type = componentDefinition.type;
        var typePrototype = type.prototype;
        var declaredInputs = {};
        var def = {
            type: type,
            providersResolver: null,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            factory: null,
            template: componentDefinition.template || null,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            hostBindings: componentDefinition.hostBindings || null,
            hostVars: componentDefinition.hostVars || 0,
            hostAttrs: componentDefinition.hostAttrs || null,
            contentQueries: componentDefinition.contentQueries || null,
            declaredInputs: declaredInputs,
            inputs: null,
            outputs: null,
            exportAs: componentDefinition.exportAs || null,
            onChanges: null,
            onInit: typePrototype.ngOnInit || null,
            doCheck: typePrototype.ngDoCheck || null,
            afterContentInit: typePrototype.ngAfterContentInit || null,
            afterContentChecked: typePrototype.ngAfterContentChecked || null,
            afterViewInit: typePrototype.ngAfterViewInit || null,
            afterViewChecked: typePrototype.ngAfterViewChecked || null,
            onDestroy: typePrototype.ngOnDestroy || null,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            selectors: componentDefinition.selectors || EMPTY_ARRAY,
            viewQuery: componentDefinition.viewQuery || null,
            features: componentDefinition.features || null,
            data: componentDefinition.data || {},
            // TODO(misko): convert ViewEncapsulation into const enum so that it can be used directly in
            // the next line. Also `None` should be 0 not 2.
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            id: 'c',
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            setInput: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
        };
        var directiveTypes = componentDefinition.directives;
        var feature = componentDefinition.features;
        var pipeTypes = componentDefinition.pipes;
        def.id += _renderCompCount++;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach(function (fn) { return fn(def); });
        def.directiveDefs = directiveTypes ?
            function () { return (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef); } :
            null;
        def.pipeDefs = pipeTypes ?
            function () { return (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef); } :
            null;
        return def;
    });
}
/**
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    var def = type.ɵcmp;
    def.directiveDefs = function () { return directives.map(extractDirectiveDef); };
    def.pipeDefs = function () { return pipes.map(extractPipeDef); };
}
export function extractDirectiveDef(type) {
    var def = getComponentDef(type) || getDirectiveDef(type);
    if (ngDevMode && !def) {
        throw new Error("'" + type.name + "' is neither 'ComponentType' or 'DirectiveType'.");
    }
    return def;
}
export function extractPipeDef(type) {
    var def = getPipeDef(type);
    if (ngDevMode && !def) {
        throw new Error("'" + type.name + "' is not a 'PipeType'.");
    }
    return def;
}
export var autoRegisterModuleById = {};
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    var res = {
        type: def.type,
        bootstrap: def.bootstrap || EMPTY_ARRAY,
        declarations: def.declarations || EMPTY_ARRAY,
        imports: def.imports || EMPTY_ARRAY,
        exports: def.exports || EMPTY_ARRAY,
        transitiveCompileScopes: null,
        schemas: def.schemas || null,
        id: def.id || null,
    };
    if (def.id != null) {
        noSideEffects(function () {
            autoRegisterModuleById[def.id] = def.type;
        });
    }
    return res;
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(function () {
        var ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    var newLookup = {};
    for (var minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            var publicName = obj[minifiedKey];
            var declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export var ɵɵdefineDirective = ɵɵdefineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition form the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
export function getFactoryDef(type, throwNotFound) {
    var hasFactoryDef = type.hasOwnProperty(NG_FACTORY_DEF);
    if (!hasFactoryDef && throwNotFound === true && ngDevMode) {
        throw new Error("Type " + stringify(type) + " does not have '\u0275fac' property.");
    }
    return hasFactoryDef ? type[NG_FACTORY_DEF] : null;
}
export function getNgModuleDef(type, throwNotFound) {
    var ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error("Type " + stringify(type) + " does not have '\u0275mod' property.");
    }
    return ngModuleDef;
}
export function getNgLocaleIdDef(type) {
    return type[NG_LOC_ID_DEF] || null;
}
//# sourceMappingURL=data:application/json;base64,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