/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { addToArray, removeFromArray } from '../util/array_utils';
import { Services } from './types';
import { declaredViewContainer, renderNode, visitRootRenderNodes } from './util';
export function attachEmbeddedView(parentView, elementData, viewIndex, view) {
    var embeddedViews = elementData.viewContainer._embeddedViews;
    if (viewIndex === null || viewIndex === undefined) {
        viewIndex = embeddedViews.length;
    }
    view.viewContainerParent = parentView;
    addToArray(embeddedViews, viewIndex, view);
    attachProjectedView(elementData, view);
    Services.dirtyParentQueries(view);
    var prevView = viewIndex > 0 ? embeddedViews[viewIndex - 1] : null;
    renderAttachEmbeddedView(elementData, prevView, view);
}
function attachProjectedView(vcElementData, view) {
    var dvcElementData = declaredViewContainer(view);
    if (!dvcElementData || dvcElementData === vcElementData ||
        view.state & 16 /* IsProjectedView */) {
        return;
    }
    // Note: For performance reasons, we
    // - add a view to template._projectedViews only 1x throughout its lifetime,
    //   and remove it not until the view is destroyed.
    //   (hard, as when a parent view is attached/detached we would need to attach/detach all
    //    nested projected views as well, even across component boundaries).
    // - don't track the insertion order of views in the projected views array
    //   (hard, as when the views of the same template are inserted different view containers)
    view.state |= 16 /* IsProjectedView */;
    var projectedViews = dvcElementData.template._projectedViews;
    if (!projectedViews) {
        projectedViews = dvcElementData.template._projectedViews = [];
    }
    projectedViews.push(view);
    // Note: we are changing the NodeDef here as we cannot calculate
    // the fact whether a template is used for projection during compilation.
    markNodeAsProjectedTemplate(view.parent.def, view.parentNodeDef);
}
function markNodeAsProjectedTemplate(viewDef, nodeDef) {
    if (nodeDef.flags & 4 /* ProjectedTemplate */) {
        return;
    }
    viewDef.nodeFlags |= 4 /* ProjectedTemplate */;
    nodeDef.flags |= 4 /* ProjectedTemplate */;
    var parentNodeDef = nodeDef.parent;
    while (parentNodeDef) {
        parentNodeDef.childFlags |= 4 /* ProjectedTemplate */;
        parentNodeDef = parentNodeDef.parent;
    }
}
export function detachEmbeddedView(elementData, viewIndex) {
    var embeddedViews = elementData.viewContainer._embeddedViews;
    if (viewIndex == null || viewIndex >= embeddedViews.length) {
        viewIndex = embeddedViews.length - 1;
    }
    if (viewIndex < 0) {
        return null;
    }
    var view = embeddedViews[viewIndex];
    view.viewContainerParent = null;
    removeFromArray(embeddedViews, viewIndex);
    // See attachProjectedView for why we don't update projectedViews here.
    Services.dirtyParentQueries(view);
    renderDetachView(view);
    return view;
}
export function detachProjectedView(view) {
    if (!(view.state & 16 /* IsProjectedView */)) {
        return;
    }
    var dvcElementData = declaredViewContainer(view);
    if (dvcElementData) {
        var projectedViews = dvcElementData.template._projectedViews;
        if (projectedViews) {
            removeFromArray(projectedViews, projectedViews.indexOf(view));
            Services.dirtyParentQueries(view);
        }
    }
}
export function moveEmbeddedView(elementData, oldViewIndex, newViewIndex) {
    var embeddedViews = elementData.viewContainer._embeddedViews;
    var view = embeddedViews[oldViewIndex];
    removeFromArray(embeddedViews, oldViewIndex);
    if (newViewIndex == null) {
        newViewIndex = embeddedViews.length;
    }
    addToArray(embeddedViews, newViewIndex, view);
    // Note: Don't need to change projectedViews as the order in there
    // as always invalid...
    Services.dirtyParentQueries(view);
    renderDetachView(view);
    var prevView = newViewIndex > 0 ? embeddedViews[newViewIndex - 1] : null;
    renderAttachEmbeddedView(elementData, prevView, view);
    return view;
}
function renderAttachEmbeddedView(elementData, prevView, view) {
    var prevRenderNode = prevView ? renderNode(prevView, prevView.def.lastRenderRootNode) : elementData.renderElement;
    var parentNode = view.renderer.parentNode(prevRenderNode);
    var nextSibling = view.renderer.nextSibling(prevRenderNode);
    // Note: We can't check if `nextSibling` is present, as on WebWorkers it will always be!
    // However, browsers automatically do `appendChild` when there is no `nextSibling`.
    visitRootRenderNodes(view, 2 /* InsertBefore */, parentNode, nextSibling, undefined);
}
export function renderDetachView(view) {
    visitRootRenderNodes(view, 3 /* RemoveChild */, null, null, undefined);
}
//# sourceMappingURL=data:application/json;base64,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