/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { __read } from "tslib";
import { ViewEncapsulation } from '../metadata/view';
import { SecurityContext } from '../sanitization/security';
import { asElementData } from './types';
import { calcBindingFlags, checkAndUpdateBinding, dispatchEvent, elementEventFullName, getParentRenderElement, NOOP, resolveDefinition, resolveRendererType2, splitMatchedQueriesDsl, splitNamespace } from './util';
export function anchorDef(flags, matchedQueriesDsl, ngContentIndex, childCount, handleEvent, templateFactory) {
    flags |= 1 /* TypeElement */;
    var _a = splitMatchedQueriesDsl(matchedQueriesDsl), matchedQueries = _a.matchedQueries, references = _a.references, matchedQueryIds = _a.matchedQueryIds;
    var template = templateFactory ? resolveDefinition(templateFactory) : null;
    return {
        // will bet set by the view definition
        nodeIndex: -1,
        parent: null,
        renderParent: null,
        bindingIndex: -1,
        outputIndex: -1,
        // regular values
        flags: flags,
        checkIndex: -1,
        childFlags: 0,
        directChildFlags: 0,
        childMatchedQueries: 0,
        matchedQueries: matchedQueries,
        matchedQueryIds: matchedQueryIds,
        references: references,
        ngContentIndex: ngContentIndex,
        childCount: childCount,
        bindings: [],
        bindingFlags: 0,
        outputs: [],
        element: {
            ns: null,
            name: null,
            attrs: null,
            template: template,
            componentProvider: null,
            componentView: null,
            componentRendererType: null,
            publicProviders: null,
            allProviders: null,
            handleEvent: handleEvent || NOOP
        },
        provider: null,
        text: null,
        query: null,
        ngContent: null
    };
}
export function elementDef(checkIndex, flags, matchedQueriesDsl, ngContentIndex, childCount, namespaceAndName, fixedAttrs, bindings, outputs, handleEvent, componentView, componentRendererType) {
    var _a;
    if (fixedAttrs === void 0) { fixedAttrs = []; }
    if (!handleEvent) {
        handleEvent = NOOP;
    }
    var _b = splitMatchedQueriesDsl(matchedQueriesDsl), matchedQueries = _b.matchedQueries, references = _b.references, matchedQueryIds = _b.matchedQueryIds;
    var ns = null;
    var name = null;
    if (namespaceAndName) {
        _a = __read(splitNamespace(namespaceAndName), 2), ns = _a[0], name = _a[1];
    }
    bindings = bindings || [];
    var bindingDefs = [];
    for (var i = 0; i < bindings.length; i++) {
        var _c = __read(bindings[i], 3), bindingFlags = _c[0], namespaceAndName_1 = _c[1], suffixOrSecurityContext = _c[2];
        var _d = __read(splitNamespace(namespaceAndName_1), 2), ns_1 = _d[0], name_1 = _d[1];
        var securityContext = undefined;
        var suffix = undefined;
        switch (bindingFlags & 15 /* Types */) {
            case 4 /* TypeElementStyle */:
                suffix = suffixOrSecurityContext;
                break;
            case 1 /* TypeElementAttribute */:
            case 8 /* TypeProperty */:
                securityContext = suffixOrSecurityContext;
                break;
        }
        bindingDefs[i] =
            { flags: bindingFlags, ns: ns_1, name: name_1, nonMinifiedName: name_1, securityContext: securityContext, suffix: suffix };
    }
    outputs = outputs || [];
    var outputDefs = [];
    for (var i = 0; i < outputs.length; i++) {
        var _e = __read(outputs[i], 2), target = _e[0], eventName = _e[1];
        outputDefs[i] =
            { type: 0 /* ElementOutput */, target: target, eventName: eventName, propName: null };
    }
    fixedAttrs = fixedAttrs || [];
    var attrs = fixedAttrs.map(function (_a) {
        var _b = __read(_a, 2), namespaceAndName = _b[0], value = _b[1];
        var _c = __read(splitNamespace(namespaceAndName), 2), ns = _c[0], name = _c[1];
        return [ns, name, value];
    });
    componentRendererType = resolveRendererType2(componentRendererType);
    if (componentView) {
        flags |= 33554432 /* ComponentView */;
    }
    flags |= 1 /* TypeElement */;
    return {
        // will bet set by the view definition
        nodeIndex: -1,
        parent: null,
        renderParent: null,
        bindingIndex: -1,
        outputIndex: -1,
        // regular values
        checkIndex: checkIndex,
        flags: flags,
        childFlags: 0,
        directChildFlags: 0,
        childMatchedQueries: 0,
        matchedQueries: matchedQueries,
        matchedQueryIds: matchedQueryIds,
        references: references,
        ngContentIndex: ngContentIndex,
        childCount: childCount,
        bindings: bindingDefs,
        bindingFlags: calcBindingFlags(bindingDefs),
        outputs: outputDefs,
        element: {
            ns: ns,
            name: name,
            attrs: attrs,
            template: null,
            // will bet set by the view definition
            componentProvider: null,
            componentView: componentView || null,
            componentRendererType: componentRendererType,
            publicProviders: null,
            allProviders: null,
            handleEvent: handleEvent || NOOP,
        },
        provider: null,
        text: null,
        query: null,
        ngContent: null
    };
}
export function createElement(view, renderHost, def) {
    var elDef = def.element;
    var rootSelectorOrNode = view.root.selectorOrNode;
    var renderer = view.renderer;
    var el;
    if (view.parent || !rootSelectorOrNode) {
        if (elDef.name) {
            el = renderer.createElement(elDef.name, elDef.ns);
        }
        else {
            el = renderer.createComment('');
        }
        var parentEl = getParentRenderElement(view, renderHost, def);
        if (parentEl) {
            renderer.appendChild(parentEl, el);
        }
    }
    else {
        // when using native Shadow DOM, do not clear the root element contents to allow slot projection
        var preserveContent = (!!elDef.componentRendererType &&
            elDef.componentRendererType.encapsulation === ViewEncapsulation.ShadowDom);
        el = renderer.selectRootElement(rootSelectorOrNode, preserveContent);
    }
    if (elDef.attrs) {
        for (var i = 0; i < elDef.attrs.length; i++) {
            var _a = __read(elDef.attrs[i], 3), ns = _a[0], name_2 = _a[1], value = _a[2];
            renderer.setAttribute(el, name_2, value, ns);
        }
    }
    return el;
}
export function listenToElementOutputs(view, compView, def, el) {
    for (var i = 0; i < def.outputs.length; i++) {
        var output = def.outputs[i];
        var handleEventClosure = renderEventHandlerClosure(view, def.nodeIndex, elementEventFullName(output.target, output.eventName));
        var listenTarget = output.target;
        var listenerView = view;
        if (output.target === 'component') {
            listenTarget = null;
            listenerView = compView;
        }
        var disposable = listenerView.renderer.listen(listenTarget || el, output.eventName, handleEventClosure);
        view.disposables[def.outputIndex + i] = disposable;
    }
}
function renderEventHandlerClosure(view, index, eventName) {
    return function (event) { return dispatchEvent(view, index, eventName, event); };
}
export function checkAndUpdateElementInline(view, def, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9) {
    var bindLen = def.bindings.length;
    var changed = false;
    if (bindLen > 0 && checkAndUpdateElementValue(view, def, 0, v0))
        changed = true;
    if (bindLen > 1 && checkAndUpdateElementValue(view, def, 1, v1))
        changed = true;
    if (bindLen > 2 && checkAndUpdateElementValue(view, def, 2, v2))
        changed = true;
    if (bindLen > 3 && checkAndUpdateElementValue(view, def, 3, v3))
        changed = true;
    if (bindLen > 4 && checkAndUpdateElementValue(view, def, 4, v4))
        changed = true;
    if (bindLen > 5 && checkAndUpdateElementValue(view, def, 5, v5))
        changed = true;
    if (bindLen > 6 && checkAndUpdateElementValue(view, def, 6, v6))
        changed = true;
    if (bindLen > 7 && checkAndUpdateElementValue(view, def, 7, v7))
        changed = true;
    if (bindLen > 8 && checkAndUpdateElementValue(view, def, 8, v8))
        changed = true;
    if (bindLen > 9 && checkAndUpdateElementValue(view, def, 9, v9))
        changed = true;
    return changed;
}
export function checkAndUpdateElementDynamic(view, def, values) {
    var changed = false;
    for (var i = 0; i < values.length; i++) {
        if (checkAndUpdateElementValue(view, def, i, values[i]))
            changed = true;
    }
    return changed;
}
function checkAndUpdateElementValue(view, def, bindingIdx, value) {
    if (!checkAndUpdateBinding(view, def, bindingIdx, value)) {
        return false;
    }
    var binding = def.bindings[bindingIdx];
    var elData = asElementData(view, def.nodeIndex);
    var renderNode = elData.renderElement;
    var name = binding.name;
    switch (binding.flags & 15 /* Types */) {
        case 1 /* TypeElementAttribute */:
            setElementAttribute(view, binding, renderNode, binding.ns, name, value);
            break;
        case 2 /* TypeElementClass */:
            setElementClass(view, renderNode, name, value);
            break;
        case 4 /* TypeElementStyle */:
            setElementStyle(view, binding, renderNode, name, value);
            break;
        case 8 /* TypeProperty */:
            var bindView = (def.flags & 33554432 /* ComponentView */ &&
                binding.flags & 32 /* SyntheticHostProperty */) ?
                elData.componentView :
                view;
            setElementProperty(bindView, binding, renderNode, name, value);
            break;
    }
    return true;
}
function setElementAttribute(view, binding, renderNode, ns, name, value) {
    var securityContext = binding.securityContext;
    var renderValue = securityContext ? view.root.sanitizer.sanitize(securityContext, value) : value;
    renderValue = renderValue != null ? renderValue.toString() : null;
    var renderer = view.renderer;
    if (value != null) {
        renderer.setAttribute(renderNode, name, renderValue, ns);
    }
    else {
        renderer.removeAttribute(renderNode, name, ns);
    }
}
function setElementClass(view, renderNode, name, value) {
    var renderer = view.renderer;
    if (value) {
        renderer.addClass(renderNode, name);
    }
    else {
        renderer.removeClass(renderNode, name);
    }
}
function setElementStyle(view, binding, renderNode, name, value) {
    var renderValue = view.root.sanitizer.sanitize(SecurityContext.STYLE, value);
    if (renderValue != null) {
        renderValue = renderValue.toString();
        var unit = binding.suffix;
        if (unit != null) {
            renderValue = renderValue + unit;
        }
    }
    else {
        renderValue = null;
    }
    var renderer = view.renderer;
    if (renderValue != null) {
        renderer.setStyle(renderNode, name, renderValue);
    }
    else {
        renderer.removeStyle(renderNode, name);
    }
}
function setElementProperty(view, binding, renderNode, name, value) {
    var securityContext = binding.securityContext;
    var renderValue = securityContext ? view.root.sanitizer.sanitize(securityContext, value) : value;
    view.renderer.setProperty(renderNode, name, renderValue);
}
//# sourceMappingURL=data:application/json;base64,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