"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var Zone = (function (global) {
    var performance = global['performance'];
    function mark(name) { performance && performance['mark'] && performance['mark'](name); }
    function performanceMeasure(name, label) {
        performance && performance['measure'] && performance['measure'](name, label);
    }
    mark('Zone');
    // Initialize before it's accessed below.
    // __Zone_symbol_prefix global can be used to override the default zone
    // symbol prefix with a custom one if needed.
    var symbolPrefix = global['__Zone_symbol_prefix'] || '__zone_symbol__';
    function __symbol__(name) { return symbolPrefix + name; }
    var checkDuplicate = global[__symbol__('forceDuplicateZoneCheck')] === true;
    if (global['Zone']) {
        // if global['Zone'] already exists (maybe zone.js was already loaded or
        // some other lib also registered a global object named Zone), we may need
        // to throw an error, but sometimes user may not want this error.
        // For example,
        // we have two web pages, page1 includes zone.js, page2 doesn't.
        // and the 1st time user load page1 and page2, everything work fine,
        // but when user load page2 again, error occurs because global['Zone'] already exists.
        // so we add a flag to let user choose whether to throw this error or not.
        // By default, if existing Zone is from zone.js, we will not throw the error.
        if (checkDuplicate || typeof global['Zone'].__symbol__ !== 'function') {
            throw new Error('Zone already loaded.');
        }
        else {
            return global['Zone'];
        }
    }
    var Zone = /** @class */ (function () {
        function Zone(parent, zoneSpec) {
            this._parent = parent;
            this._name = zoneSpec ? zoneSpec.name || 'unnamed' : '<root>';
            this._properties = zoneSpec && zoneSpec.properties || {};
            this._zoneDelegate =
                new ZoneDelegate(this, this._parent && this._parent._zoneDelegate, zoneSpec);
        }
        Zone.assertZonePatched = function () {
            if (global['Promise'] !== patches['ZoneAwarePromise']) {
                throw new Error('Zone.js has detected that ZoneAwarePromise `(window|global).Promise` ' +
                    'has been overwritten.\n' +
                    'Most likely cause is that a Promise polyfill has been loaded ' +
                    'after Zone.js (Polyfilling Promise api is not necessary when zone.js is loaded. ' +
                    'If you must load one, do so before loading zone.js.)');
            }
        };
        Object.defineProperty(Zone, "root", {
            get: function () {
                var zone = Zone.current;
                while (zone.parent) {
                    zone = zone.parent;
                }
                return zone;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Zone, "current", {
            get: function () { return _currentZoneFrame.zone; },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Zone, "currentTask", {
            get: function () { return _currentTask; },
            enumerable: true,
            configurable: true
        });
        // tslint:disable-next-line:require-internal-with-underscore
        Zone.__load_patch = function (name, fn) {
            if (patches.hasOwnProperty(name)) {
                if (checkDuplicate) {
                    throw Error('Already loaded patch: ' + name);
                }
            }
            else if (!global['__Zone_disable_' + name]) {
                var perfName = 'Zone:' + name;
                mark(perfName);
                patches[name] = fn(global, Zone, _api);
                performanceMeasure(perfName, perfName);
            }
        };
        Object.defineProperty(Zone.prototype, "parent", {
            get: function () { return this._parent; },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Zone.prototype, "name", {
            get: function () { return this._name; },
            enumerable: true,
            configurable: true
        });
        Zone.prototype.get = function (key) {
            var zone = this.getZoneWith(key);
            if (zone)
                return zone._properties[key];
        };
        Zone.prototype.getZoneWith = function (key) {
            var current = this;
            while (current) {
                if (current._properties.hasOwnProperty(key)) {
                    return current;
                }
                current = current._parent;
            }
            return null;
        };
        Zone.prototype.fork = function (zoneSpec) {
            if (!zoneSpec)
                throw new Error('ZoneSpec required!');
            return this._zoneDelegate.fork(this, zoneSpec);
        };
        Zone.prototype.wrap = function (callback, source) {
            if (typeof callback !== 'function') {
                throw new Error('Expecting function got: ' + callback);
            }
            var _callback = this._zoneDelegate.intercept(this, callback, source);
            var zone = this;
            return function () {
                return zone.runGuarded(_callback, this, arguments, source);
            };
        };
        Zone.prototype.run = function (callback, applyThis, applyArgs, source) {
            _currentZoneFrame = { parent: _currentZoneFrame, zone: this };
            try {
                return this._zoneDelegate.invoke(this, callback, applyThis, applyArgs, source);
            }
            finally {
                _currentZoneFrame = _currentZoneFrame.parent;
            }
        };
        Zone.prototype.runGuarded = function (callback, applyThis, applyArgs, source) {
            if (applyThis === void 0) { applyThis = null; }
            _currentZoneFrame = { parent: _currentZoneFrame, zone: this };
            try {
                try {
                    return this._zoneDelegate.invoke(this, callback, applyThis, applyArgs, source);
                }
                catch (error) {
                    if (this._zoneDelegate.handleError(this, error)) {
                        throw error;
                    }
                }
            }
            finally {
                _currentZoneFrame = _currentZoneFrame.parent;
            }
        };
        Zone.prototype.runTask = function (task, applyThis, applyArgs) {
            if (task.zone != this) {
                throw new Error('A task can only be run in the zone of creation! (Creation: ' +
                    (task.zone || NO_ZONE).name + '; Execution: ' + this.name + ')');
            }
            // https://github.com/angular/zone.js/issues/778, sometimes eventTask
            // will run in notScheduled(canceled) state, we should not try to
            // run such kind of task but just return
            if (task.state === notScheduled && (task.type === eventTask || task.type === macroTask)) {
                return;
            }
            var reEntryGuard = task.state != running;
            reEntryGuard && task._transitionTo(running, scheduled);
            task.runCount++;
            var previousTask = _currentTask;
            _currentTask = task;
            _currentZoneFrame = { parent: _currentZoneFrame, zone: this };
            try {
                if (task.type == macroTask && task.data && !task.data.isPeriodic) {
                    task.cancelFn = undefined;
                }
                try {
                    return this._zoneDelegate.invokeTask(this, task, applyThis, applyArgs);
                }
                catch (error) {
                    if (this._zoneDelegate.handleError(this, error)) {
                        throw error;
                    }
                }
            }
            finally {
                // if the task's state is notScheduled or unknown, then it has already been cancelled
                // we should not reset the state to scheduled
                if (task.state !== notScheduled && task.state !== unknown) {
                    if (task.type == eventTask || (task.data && task.data.isPeriodic)) {
                        reEntryGuard && task._transitionTo(scheduled, running);
                    }
                    else {
                        task.runCount = 0;
                        this._updateTaskCount(task, -1);
                        reEntryGuard &&
                            task._transitionTo(notScheduled, running, notScheduled);
                    }
                }
                _currentZoneFrame = _currentZoneFrame.parent;
                _currentTask = previousTask;
            }
        };
        Zone.prototype.scheduleTask = function (task) {
            if (task.zone && task.zone !== this) {
                // check if the task was rescheduled, the newZone
                // should not be the children of the original zone
                var newZone = this;
                while (newZone) {
                    if (newZone === task.zone) {
                        throw Error("can not reschedule task to " + this.name + " which is descendants of the original zone " + task.zone.name);
                    }
                    newZone = newZone.parent;
                }
            }
            task._transitionTo(scheduling, notScheduled);
            var zoneDelegates = [];
            task._zoneDelegates = zoneDelegates;
            task._zone = this;
            try {
                task = this._zoneDelegate.scheduleTask(this, task);
            }
            catch (err) {
                // should set task's state to unknown when scheduleTask throw error
                // because the err may from reschedule, so the fromState maybe notScheduled
                task._transitionTo(unknown, scheduling, notScheduled);
                // TODO: @JiaLiPassion, should we check the result from handleError?
                this._zoneDelegate.handleError(this, err);
                throw err;
            }
            if (task._zoneDelegates === zoneDelegates) {
                // we have to check because internally the delegate can reschedule the task.
                this._updateTaskCount(task, 1);
            }
            if (task.state == scheduling) {
                task._transitionTo(scheduled, scheduling);
            }
            return task;
        };
        Zone.prototype.scheduleMicroTask = function (source, callback, data, customSchedule) {
            return this.scheduleTask(new ZoneTask(microTask, source, callback, data, customSchedule, undefined));
        };
        Zone.prototype.scheduleMacroTask = function (source, callback, data, customSchedule, customCancel) {
            return this.scheduleTask(new ZoneTask(macroTask, source, callback, data, customSchedule, customCancel));
        };
        Zone.prototype.scheduleEventTask = function (source, callback, data, customSchedule, customCancel) {
            return this.scheduleTask(new ZoneTask(eventTask, source, callback, data, customSchedule, customCancel));
        };
        Zone.prototype.cancelTask = function (task) {
            if (task.zone != this)
                throw new Error('A task can only be cancelled in the zone of creation! (Creation: ' +
                    (task.zone || NO_ZONE).name + '; Execution: ' + this.name + ')');
            task._transitionTo(canceling, scheduled, running);
            try {
                this._zoneDelegate.cancelTask(this, task);
            }
            catch (err) {
                // if error occurs when cancelTask, transit the state to unknown
                task._transitionTo(unknown, canceling);
                this._zoneDelegate.handleError(this, err);
                throw err;
            }
            this._updateTaskCount(task, -1);
            task._transitionTo(notScheduled, canceling);
            task.runCount = 0;
            return task;
        };
        Zone.prototype._updateTaskCount = function (task, count) {
            var zoneDelegates = task._zoneDelegates;
            if (count == -1) {
                task._zoneDelegates = null;
            }
            for (var i = 0; i < zoneDelegates.length; i++) {
                zoneDelegates[i]._updateTaskCount(task.type, count);
            }
        };
        // tslint:disable-next-line:require-internal-with-underscore
        Zone.__symbol__ = __symbol__;
        return Zone;
    }());
    var DELEGATE_ZS = {
        name: '',
        onHasTask: function (delegate, _, target, hasTaskState) { return delegate.hasTask(target, hasTaskState); },
        onScheduleTask: function (delegate, _, target, task) { return delegate.scheduleTask(target, task); },
        onInvokeTask: function (delegate, _, target, task, applyThis, applyArgs) {
            return delegate.invokeTask(target, task, applyThis, applyArgs);
        },
        onCancelTask: function (delegate, _, target, task) { return delegate.cancelTask(target, task); }
    };
    var ZoneDelegate = /** @class */ (function () {
        function ZoneDelegate(zone, parentDelegate, zoneSpec) {
            this._taskCounts = { 'microTask': 0, 'macroTask': 0, 'eventTask': 0 };
            this.zone = zone;
            this._parentDelegate = parentDelegate;
            this._forkZS =
                zoneSpec && (zoneSpec && zoneSpec.onFork ? zoneSpec : parentDelegate._forkZS);
            this._forkDlgt = zoneSpec && (zoneSpec.onFork ? parentDelegate : parentDelegate._forkDlgt);
            this._forkCurrZone =
                zoneSpec && (zoneSpec.onFork ? this.zone : parentDelegate._forkCurrZone);
            this._interceptZS =
                zoneSpec && (zoneSpec.onIntercept ? zoneSpec : parentDelegate._interceptZS);
            this._interceptDlgt =
                zoneSpec && (zoneSpec.onIntercept ? parentDelegate : parentDelegate._interceptDlgt);
            this._interceptCurrZone =
                zoneSpec && (zoneSpec.onIntercept ? this.zone : parentDelegate._interceptCurrZone);
            this._invokeZS = zoneSpec && (zoneSpec.onInvoke ? zoneSpec : parentDelegate._invokeZS);
            this._invokeDlgt =
                zoneSpec && (zoneSpec.onInvoke ? parentDelegate : parentDelegate._invokeDlgt);
            this._invokeCurrZone =
                zoneSpec && (zoneSpec.onInvoke ? this.zone : parentDelegate._invokeCurrZone);
            this._handleErrorZS =
                zoneSpec && (zoneSpec.onHandleError ? zoneSpec : parentDelegate._handleErrorZS);
            this._handleErrorDlgt = zoneSpec &&
                (zoneSpec.onHandleError ? parentDelegate : parentDelegate._handleErrorDlgt);
            this._handleErrorCurrZone =
                zoneSpec && (zoneSpec.onHandleError ? this.zone : parentDelegate._handleErrorCurrZone);
            this._scheduleTaskZS =
                zoneSpec && (zoneSpec.onScheduleTask ? zoneSpec : parentDelegate._scheduleTaskZS);
            this._scheduleTaskDlgt = zoneSpec &&
                (zoneSpec.onScheduleTask ? parentDelegate : parentDelegate._scheduleTaskDlgt);
            this._scheduleTaskCurrZone = zoneSpec &&
                (zoneSpec.onScheduleTask ? this.zone : parentDelegate._scheduleTaskCurrZone);
            this._invokeTaskZS =
                zoneSpec && (zoneSpec.onInvokeTask ? zoneSpec : parentDelegate._invokeTaskZS);
            this._invokeTaskDlgt =
                zoneSpec && (zoneSpec.onInvokeTask ? parentDelegate : parentDelegate._invokeTaskDlgt);
            this._invokeTaskCurrZone =
                zoneSpec && (zoneSpec.onInvokeTask ? this.zone : parentDelegate._invokeTaskCurrZone);
            this._cancelTaskZS =
                zoneSpec && (zoneSpec.onCancelTask ? zoneSpec : parentDelegate._cancelTaskZS);
            this._cancelTaskDlgt =
                zoneSpec && (zoneSpec.onCancelTask ? parentDelegate : parentDelegate._cancelTaskDlgt);
            this._cancelTaskCurrZone =
                zoneSpec && (zoneSpec.onCancelTask ? this.zone : parentDelegate._cancelTaskCurrZone);
            this._hasTaskZS = null;
            this._hasTaskDlgt = null;
            this._hasTaskDlgtOwner = null;
            this._hasTaskCurrZone = null;
            var zoneSpecHasTask = zoneSpec && zoneSpec.onHasTask;
            var parentHasTask = parentDelegate && parentDelegate._hasTaskZS;
            if (zoneSpecHasTask || parentHasTask) {
                // If we need to report hasTask, than this ZS needs to do ref counting on tasks. In such
                // a case all task related interceptors must go through this ZD. We can't short circuit it.
                this._hasTaskZS = zoneSpecHasTask ? zoneSpec : DELEGATE_ZS;
                this._hasTaskDlgt = parentDelegate;
                this._hasTaskDlgtOwner = this;
                this._hasTaskCurrZone = zone;
                if (!zoneSpec.onScheduleTask) {
                    this._scheduleTaskZS = DELEGATE_ZS;
                    this._scheduleTaskDlgt = parentDelegate;
                    this._scheduleTaskCurrZone = this.zone;
                }
                if (!zoneSpec.onInvokeTask) {
                    this._invokeTaskZS = DELEGATE_ZS;
                    this._invokeTaskDlgt = parentDelegate;
                    this._invokeTaskCurrZone = this.zone;
                }
                if (!zoneSpec.onCancelTask) {
                    this._cancelTaskZS = DELEGATE_ZS;
                    this._cancelTaskDlgt = parentDelegate;
                    this._cancelTaskCurrZone = this.zone;
                }
            }
        }
        ZoneDelegate.prototype.fork = function (targetZone, zoneSpec) {
            return this._forkZS ?
                this._forkZS.onFork(this._forkDlgt, this.zone, targetZone, zoneSpec) :
                new Zone(targetZone, zoneSpec);
        };
        ZoneDelegate.prototype.intercept = function (targetZone, callback, source) {
            return this._interceptZS ?
                this._interceptZS.onIntercept(this._interceptDlgt, this._interceptCurrZone, targetZone, callback, source) :
                callback;
        };
        ZoneDelegate.prototype.invoke = function (targetZone, callback, applyThis, applyArgs, source) {
            return this._invokeZS ?
                this._invokeZS.onInvoke(this._invokeDlgt, this._invokeCurrZone, targetZone, callback, applyThis, applyArgs, source) :
                callback.apply(applyThis, applyArgs);
        };
        ZoneDelegate.prototype.handleError = function (targetZone, error) {
            return this._handleErrorZS ?
                this._handleErrorZS.onHandleError(this._handleErrorDlgt, this._handleErrorCurrZone, targetZone, error) :
                true;
        };
        ZoneDelegate.prototype.scheduleTask = function (targetZone, task) {
            var returnTask = task;
            if (this._scheduleTaskZS) {
                if (this._hasTaskZS) {
                    returnTask._zoneDelegates.push(this._hasTaskDlgtOwner);
                }
                // clang-format off
                returnTask = this._scheduleTaskZS.onScheduleTask(this._scheduleTaskDlgt, this._scheduleTaskCurrZone, targetZone, task);
                // clang-format on
                if (!returnTask)
                    returnTask = task;
            }
            else {
                if (task.scheduleFn) {
                    task.scheduleFn(task);
                }
                else if (task.type == microTask) {
                    scheduleMicroTask(task);
                }
                else {
                    throw new Error('Task is missing scheduleFn.');
                }
            }
            return returnTask;
        };
        ZoneDelegate.prototype.invokeTask = function (targetZone, task, applyThis, applyArgs) {
            return this._invokeTaskZS ?
                this._invokeTaskZS.onInvokeTask(this._invokeTaskDlgt, this._invokeTaskCurrZone, targetZone, task, applyThis, applyArgs) :
                task.callback.apply(applyThis, applyArgs);
        };
        ZoneDelegate.prototype.cancelTask = function (targetZone, task) {
            var value;
            if (this._cancelTaskZS) {
                value = this._cancelTaskZS.onCancelTask(this._cancelTaskDlgt, this._cancelTaskCurrZone, targetZone, task);
            }
            else {
                if (!task.cancelFn) {
                    throw Error('Task is not cancelable');
                }
                value = task.cancelFn(task);
            }
            return value;
        };
        ZoneDelegate.prototype.hasTask = function (targetZone, isEmpty) {
            // hasTask should not throw error so other ZoneDelegate
            // can still trigger hasTask callback
            try {
                this._hasTaskZS &&
                    this._hasTaskZS.onHasTask(this._hasTaskDlgt, this._hasTaskCurrZone, targetZone, isEmpty);
            }
            catch (err) {
                this.handleError(targetZone, err);
            }
        };
        // tslint:disable-next-line:require-internal-with-underscore
        ZoneDelegate.prototype._updateTaskCount = function (type, count) {
            var counts = this._taskCounts;
            var prev = counts[type];
            var next = counts[type] = prev + count;
            if (next < 0) {
                throw new Error('More tasks executed then were scheduled.');
            }
            if (prev == 0 || next == 0) {
                var isEmpty = {
                    microTask: counts['microTask'] > 0,
                    macroTask: counts['macroTask'] > 0,
                    eventTask: counts['eventTask'] > 0,
                    change: type
                };
                this.hasTask(this.zone, isEmpty);
            }
        };
        return ZoneDelegate;
    }());
    var ZoneTask = /** @class */ (function () {
        function ZoneTask(type, source, callback, options, scheduleFn, cancelFn) {
            // tslint:disable-next-line:require-internal-with-underscore
            this._zone = null;
            this.runCount = 0;
            // tslint:disable-next-line:require-internal-with-underscore
            this._zoneDelegates = null;
            // tslint:disable-next-line:require-internal-with-underscore
            this._state = 'notScheduled';
            this.type = type;
            this.source = source;
            this.data = options;
            this.scheduleFn = scheduleFn;
            this.cancelFn = cancelFn;
            if (!callback) {
                throw new Error('callback is not defined');
            }
            this.callback = callback;
            var self = this;
            // TODO: @JiaLiPassion options should have interface
            if (type === eventTask && options && options.useG) {
                this.invoke = ZoneTask.invokeTask;
            }
            else {
                this.invoke = function () {
                    return ZoneTask.invokeTask.call(global, self, this, arguments);
                };
            }
        }
        ZoneTask.invokeTask = function (task, target, args) {
            if (!task) {
                task = this;
            }
            _numberOfNestedTaskFrames++;
            try {
                task.runCount++;
                return task.zone.runTask(task, target, args);
            }
            finally {
                if (_numberOfNestedTaskFrames == 1) {
                    drainMicroTaskQueue();
                }
                _numberOfNestedTaskFrames--;
            }
        };
        Object.defineProperty(ZoneTask.prototype, "zone", {
            get: function () { return this._zone; },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(ZoneTask.prototype, "state", {
            get: function () { return this._state; },
            enumerable: true,
            configurable: true
        });
        ZoneTask.prototype.cancelScheduleRequest = function () { this._transitionTo(notScheduled, scheduling); };
        // tslint:disable-next-line:require-internal-with-underscore
        ZoneTask.prototype._transitionTo = function (toState, fromState1, fromState2) {
            if (this._state === fromState1 || this._state === fromState2) {
                this._state = toState;
                if (toState == notScheduled) {
                    this._zoneDelegates = null;
                }
            }
            else {
                throw new Error(this.type + " '" + this.source + "': can not transition to '" + toState + "', expecting state '" + fromState1 + "'" + (fromState2 ? ' or \'' + fromState2 + '\'' : '') + ", was '" + this._state + "'.");
            }
        };
        ZoneTask.prototype.toString = function () {
            if (this.data && typeof this.data.handleId !== 'undefined') {
                return this.data.handleId.toString();
            }
            else {
                return Object.prototype.toString.call(this);
            }
        };
        // add toJSON method to prevent cyclic error when
        // call JSON.stringify(zoneTask)
        ZoneTask.prototype.toJSON = function () {
            return {
                type: this.type,
                state: this.state,
                source: this.source,
                zone: this.zone.name,
                runCount: this.runCount
            };
        };
        return ZoneTask;
    }());
    //////////////////////////////////////////////////////
    //////////////////////////////////////////////////////
    ///  MICROTASK QUEUE
    //////////////////////////////////////////////////////
    //////////////////////////////////////////////////////
    var symbolSetTimeout = __symbol__('setTimeout');
    var symbolPromise = __symbol__('Promise');
    var symbolThen = __symbol__('then');
    var _microTaskQueue = [];
    var _isDrainingMicrotaskQueue = false;
    var nativeMicroTaskQueuePromise;
    function scheduleMicroTask(task) {
        // if we are not running in any task, and there has not been anything scheduled
        // we must bootstrap the initial task creation by manually scheduling the drain
        if (_numberOfNestedTaskFrames === 0 && _microTaskQueue.length === 0) {
            // We are not running in Task, so we need to kickstart the microtask queue.
            if (!nativeMicroTaskQueuePromise) {
                if (global[symbolPromise]) {
                    nativeMicroTaskQueuePromise = global[symbolPromise].resolve(0);
                }
            }
            if (nativeMicroTaskQueuePromise) {
                var nativeThen = nativeMicroTaskQueuePromise[symbolThen];
                if (!nativeThen) {
                    // native Promise is not patchable, we need to use `then` directly
                    // issue 1078
                    nativeThen = nativeMicroTaskQueuePromise['then'];
                }
                nativeThen.call(nativeMicroTaskQueuePromise, drainMicroTaskQueue);
            }
            else {
                global[symbolSetTimeout](drainMicroTaskQueue, 0);
            }
        }
        task && _microTaskQueue.push(task);
    }
    function drainMicroTaskQueue() {
        if (!_isDrainingMicrotaskQueue) {
            _isDrainingMicrotaskQueue = true;
            while (_microTaskQueue.length) {
                var queue = _microTaskQueue;
                _microTaskQueue = [];
                for (var i = 0; i < queue.length; i++) {
                    var task = queue[i];
                    try {
                        task.zone.runTask(task, null, null);
                    }
                    catch (error) {
                        _api.onUnhandledError(error);
                    }
                }
            }
            _api.microtaskDrainDone();
            _isDrainingMicrotaskQueue = false;
        }
    }
    //////////////////////////////////////////////////////
    //////////////////////////////////////////////////////
    ///  BOOTSTRAP
    //////////////////////////////////////////////////////
    //////////////////////////////////////////////////////
    var NO_ZONE = { name: 'NO ZONE' };
    var notScheduled = 'notScheduled', scheduling = 'scheduling', scheduled = 'scheduled', running = 'running', canceling = 'canceling', unknown = 'unknown';
    var microTask = 'microTask', macroTask = 'macroTask', eventTask = 'eventTask';
    var patches = {};
    var _api = {
        symbol: __symbol__,
        currentZoneFrame: function () { return _currentZoneFrame; },
        onUnhandledError: noop,
        microtaskDrainDone: noop,
        scheduleMicroTask: scheduleMicroTask,
        showUncaughtError: function () { return !Zone[__symbol__('ignoreConsoleErrorUncaughtError')]; },
        patchEventTarget: function () { return []; },
        patchOnProperties: noop,
        patchMethod: function () { return noop; },
        bindArguments: function () { return []; },
        patchThen: function () { return noop; },
        patchMacroTask: function () { return noop; },
        setNativePromise: function (NativePromise) {
            // sometimes NativePromise.resolve static function
            // is not ready yet, (such as core-js/es6.promise)
            // so we need to check here.
            if (NativePromise && typeof NativePromise.resolve === 'function') {
                nativeMicroTaskQueuePromise = NativePromise.resolve(0);
            }
        },
        patchEventPrototype: function () { return noop; },
        isIEOrEdge: function () { return false; },
        getGlobalObjects: function () { return undefined; },
        ObjectDefineProperty: function () { return noop; },
        ObjectGetOwnPropertyDescriptor: function () { return undefined; },
        ObjectCreate: function () { return undefined; },
        ArraySlice: function () { return []; },
        patchClass: function () { return noop; },
        wrapWithCurrentZone: function () { return noop; },
        filterProperties: function () { return []; },
        attachOriginToPatched: function () { return noop; },
        _redefineProperty: function () { return noop; },
        patchCallbacks: function () { return noop; }
    };
    var _currentZoneFrame = { parent: null, zone: new Zone(null, null) };
    var _currentTask = null;
    var _numberOfNestedTaskFrames = 0;
    function noop() { }
    performanceMeasure('Zone', 'Zone');
    return global['Zone'] = Zone;
})(typeof window !== 'undefined' && window || typeof self !== 'undefined' && self || global);
//# sourceMappingURL=data:application/json;base64,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