package main

import (
	"context"
	"flag"
	"fmt"
	"github.com/go-kit/kit/log"
	"github.com/joho/godotenv"
	"go.opentelemetry.io/otel"
	"go.opentelemetry.io/otel/exporters/otlp/otlptrace/otlptracehttp"
	"go.opentelemetry.io/otel/exporters/stdout/stdouttrace"
	"go.opentelemetry.io/otel/sdk/resource"
	oteltrace "go.opentelemetry.io/otel/sdk/trace"
	sdktrace "go.opentelemetry.io/otel/sdk/trace"
	semconv "go.opentelemetry.io/otel/semconv/v1.24.0"
	"go.opentelemetry.io/otel/trace"
	"google.golang.org/grpc"
	log2 "log"
	"net"
	"net/http"
	"os"
	"otel_microservice_test/endpoint"
	"otel_microservice_test/service"
	"otel_microservice_test/transport"
	"time"
)

var (
	tracer       trace.Tracer
	otlpEndpoint string
)

func init() {
	godotenv.Load()
	otlpEndpoint = os.Getenv("OTLP_ENDPOINT")
	if otlpEndpoint == "" {
		log2.Fatalln("You MUST set OTLP_ENDPOINT env variable!")
	}
}

func newConsoleExporter() (oteltrace.SpanExporter, error) {
	return stdouttrace.New()
}

func newOTLPExporter(ctx context.Context) (oteltrace.SpanExporter, error) {
	insecureOpt := otlptracehttp.WithInsecure()

	endpointOpt := otlptracehttp.WithEndpoint(otlpEndpoint)

	return otlptracehttp.New(ctx, insecureOpt, endpointOpt)
}

func newTraceProvider(exp sdktrace.SpanExporter) *sdktrace.TracerProvider {
	r, err := resource.Merge(
		resource.Default(),
		resource.NewWithAttributes(
			semconv.SchemaURL,
			semconv.ServiceName("service_1"),
		),
	)

	if err != nil {
		panic(err)
	}

	return sdktrace.NewTracerProvider(
		sdktrace.WithBatcher(exp),
		sdktrace.WithResource(r),
	)
}

func Start() (net.Listener, error) {
	ctx := context.Background()

	//exp, err := newConsoleExporter()
	exp, err := newOTLPExporter(ctx)

	if err != nil {
		log2.Fatalf("failed to initialize exporter: %v", err)
	}

	tp := newTraceProvider(exp)

	defer func() { _ = tp.Shutdown(ctx) }()

	otel.SetTracerProvider(tp)
	tracer = tp.Tracer("service_1")
	httpAddr := flag.String("httpAddr", ":8081", "HTTP listen address")
	grpcAddr := flag.String("grpcAddr", ":9085", "GRPC listen address")
	flag.Parse()
	fmt.Println("httpAddr", *httpAddr)
	fmt.Println("grpcAddr", *grpcAddr)

	var logger log.Logger
	logger = log.NewLogfmtLogger(os.Stderr)
	logger = log.With(logger, "caller", log.DefaultCaller)

	var logSvc service.LogService
	{
		conn, err := grpc.Dial(*grpcAddr, grpc.WithInsecure(), grpc.WithTimeout(time.Second))
		if err != nil {
			logger.Log("during", "connecting proxy", "error", err)
			os.Exit(1)
		}
		defer conn.Close()
		logSvc = transport.NewGRPCClient(conn, tracer)
	}

	var (
		svc         = service.NewService(tracer)
		endpoints   = endpoint.MakeServerEndpoint(svc, tracer)
		logEndpoint = endpoint.MakeServerLogEndpoint(logSvc, tracer)
		httpHandler = transport.NewHTTPHandler(endpoints, logEndpoint, logger, tracer)
	)

	httpListener, err := net.Listen("tcp", *httpAddr)
	if err != nil {
		logger.Log("transport", "HTTP", "during", "Listen", "err", err)
		os.Exit(1)
		return httpListener, err
	}
	logger.Log("transport", "Http", "addr", *httpAddr)
	return httpListener, http.Serve(httpListener, httpHandler)
}

func main() {
	httpListener, err := Start()
	if err != nil {
		httpListener.Close()
	}
}
