package transport

import (
	"context"
	"encoding/json"
	"github.com/go-kit/kit/endpoint"
	log2 "github.com/go-kit/kit/log"
	"github.com/go-kit/kit/transport"
	httpTransport "github.com/go-kit/kit/transport/http"
	"go.opentelemetry.io/otel/trace"
	"net/http"
	endpoint2 "otel_microservice_test/endpoint"
	"otel_microservice_test/service"
)

func NewHTTPHandler(endpoints endpoint2.TestEndpoint, logEndpoint endpoint2.LogEndpoints, logger log2.Logger, tracer trace.Tracer) http.Handler {
	mux := http.NewServeMux()

	options := []httpTransport.ServerOption{
		httpTransport.ServerErrorEncoder(errorEncoder),
		httpTransport.ServerErrorHandler(transport.NewLogErrorHandler(logger)),
	}

	mux.Handle("/test", httpTransport.NewServer(
		endpoints.TestEndpoint,
		decodeTestReq(tracer),
		encodeTestRes,
		options...,
	))

	mux.Handle("/log", httpTransport.NewServer(
		logEndpoint.NewLogEndpoint,
		decodeHTTPLogReq,
		encodeHTTPLogRes,
		options...,
	))

	return mux
}

// ================================ //DECODER// ================================ //

func decodeTestReq(tracer trace.Tracer) httpTransport.DecodeRequestFunc {
	return func(_ context.Context, r *http.Request) (interface{}, error) {
		return nil, nil
	}
}

func decodeHTTPLogReq(_ context.Context, r *http.Request) (interface{}, error) {
	var req service.LogReq
	err := json.NewDecoder(r.Body).Decode(&req)
	return req, err
}

// ================================ //ENCODER// ================================ //

func encodeTestRes(ctx context.Context, w http.ResponseWriter, response interface{}) error {
	if f, ok := response.(endpoint.Failer); ok && f.Failed() != nil {
		errorEncoder(ctx, f.Failed(), w)
		return nil
	}
	return json.NewEncoder(w).Encode(response)
}

func encodeHTTPLogRes(ctx context.Context, w http.ResponseWriter, response interface{}) error {
	if f, ok := response.(endpoint.Failer); ok && f.Failed() != nil {
		errorEncoder(ctx, f.Failed(), w)
		return nil
	}
	return json.NewEncoder(w).Encode(response)
}

func errorEncoder(_ context.Context, err error, w http.ResponseWriter) {
	w.WriteHeader(http.StatusInternalServerError)
	json.NewEncoder(w).Encode(errorWrapper{
		Error: err.Error(),
		Code:  500,
	})
}

type errorWrapper struct {
	Error string `json:"error"`
	Code  int    `json:"code"`
}
